/*
This file is part of Telegram Desktop,
the official desktop application for the Telegram messaging service.

For license and copyright information please follow this link:
https://github.com/telegramdesktop/tdesktop/blob/master/LEGAL
*/
#include "history/view/media/history_view_service_box.h"

#include "core/ui_integration.h"
#include "data/data_session.h"
#include "history/view/media/history_view_sticker_player_abstract.h"
#include "history/view/history_view_cursor_state.h"
#include "history/view/history_view_element.h"
#include "history/view/history_view_text_helper.h"
#include "history/history.h"
#include "history/history_item.h"
#include "lang/lang_keys.h"
#include "ui/chat/chat_style.h"
#include "ui/effects/animation_value.h"
#include "ui/effects/premium_stars_colored.h"
#include "ui/effects/ripple_animation.h"
#include "ui/text/text_utilities.h"
#include "ui/painter.h"
#include "ui/rect.h"
#include "ui/power_saving.h"
#include "styles/style_chat.h"
#include "styles/style_credits.h"
#include "styles/style_premium.h"
#include "styles/style_layers.h"

namespace HistoryView {

int ServiceBoxContent::width() {
	return st::msgServiceGiftBoxSize.width();
}

ServiceBox::ServiceBox(
	not_null<Element*> parent,
	std::unique_ptr<ServiceBoxContent> content)
: Media(parent)
, _parent(parent)
, _content(std::move(content))
, _button({ .link = _content->createViewLink() })
, _maxWidth(_content->width()
	- st::msgPadding.left()
	- st::msgPadding.right())
, _title(
	st::defaultSubsectionTitle.style,
	_content->title(),
	kMarkupTextOptions,
	_maxWidth,
	Core::TextContext({
		.session = &parent->history()->session(),
		.repaint = [parent] { parent->customEmojiRepaint(); },
	}))
, _author(
	st::uniqueGiftReleasedBy.style,
	_content->author(),
	kMarkupTextOptions,
	_maxWidth)
, _subtitle(
	st::premiumPreviewAbout.style,
	Ui::Text::Filtered(
		_content->subtitle(),
		{
			EntityType::Bold,
			EntityType::StrikeOut,
			EntityType::Underline,
			EntityType::Italic,
			EntityType::Spoiler,
			EntityType::CustomEmoji,
		}),
	kMarkupTextOptions,
	_maxWidth,
	Core::TextContext({
		.session = &parent->history()->session(),
		.repaint = [parent] { parent->customEmojiRepaint(); },
	}))
, _size(
	_content->width(),
	(st::msgServiceGiftBoxTopSkip
		+ _content->top()
		+ _content->size().height()
		+ st::msgServiceGiftBoxTitlePadding.top()
		+ (_title.isEmpty()
			? 0
			: (_title.countHeight(_maxWidth)
				+ st::msgServiceGiftBoxTitlePadding.bottom()))
		+ (_author.isEmpty()
			? 0
			: (st::giftBoxReleasedByMargin.top()
				+ st::uniqueGiftReleasedBy.style.font->height
				+ st::giftBoxReleasedByMargin.bottom()
				+ st::msgServiceGiftBoxTitlePadding.bottom()))
		+ _subtitle.countHeight(_maxWidth)
		+ (!_content->button()
			? 0
			: (_content->buttonSkip() + st::msgServiceGiftBoxButtonHeight))
		+ st::msgServiceGiftBoxButtonMargins.bottom()))
, _innerSize(_size - QSize(0, st::msgServiceGiftBoxTopSkip)) {
	InitElementTextPart(_parent, _subtitle);
	if (auto text = _content->button()) {
		_button.repaint = [=] { repaint(); };
		std::move(text) | rpl::start_with_next([=](QString value) {
			_button.text.setText(st::semiboldTextStyle, value);
			const auto height = st::msgServiceGiftBoxButtonHeight;
			const auto &padding = st::msgServiceGiftBoxButtonPadding;
			const auto empty = _button.size.isEmpty();
			_button.size = QSize(
				(_button.text.maxWidth()
					+ height
					+ padding.left()
					+ padding.right()),
				height);
			if (!empty) {
				repaint();
			}
		}, _lifetime);
	}
	if (const auto type = _content->buttonMinistars()) {
		_button.stars = std::make_unique<Ui::Premium::ColoredMiniStars>(
			[=](const QRect &) { repaint(); },
			*type);
		_button.lastFg = std::make_unique<QColor>();
	}

	if (auto changes = _content->changes()) {
		std::move(changes) | rpl::start_with_next([=] {
			applyContentChanges();
		}, _lifetime);
	}
}

ServiceBox::~ServiceBox() = default;

void ServiceBox::applyContentChanges() {
	const auto subtitleWas = _subtitle.countHeight(_maxWidth);

	const auto parent = _parent;
	_subtitle = Ui::Text::String(
		st::premiumPreviewAbout.style,
		Ui::Text::Filtered(
			_content->subtitle(),
			{
				EntityType::Bold,
				EntityType::StrikeOut,
				EntityType::Underline,
				EntityType::Italic,
				EntityType::Spoiler,
				EntityType::CustomEmoji,
			}),
			kMarkupTextOptions,
			_maxWidth,
			Core::TextContext({
				.session = &parent->history()->session(),
				.repaint = [parent] { parent->customEmojiRepaint(); },
			}));
	InitElementTextPart(parent, _subtitle);
	const auto subtitleNow = _subtitle.countHeight(_maxWidth);
	if (subtitleNow != subtitleWas) {
		_size.setHeight(_size.height() - subtitleWas + subtitleNow);
		_innerSize = _size - QSize(0, st::msgServiceGiftBoxTopSkip);

		const auto item = parent->data();
		item->history()->owner().requestItemResize(item);
	} else {
		parent->repaint();
	}
}

QSize ServiceBox::countOptimalSize() {
	return _size;
}

QSize ServiceBox::countCurrentSize(int newWidth) {
	return _size;
}

void ServiceBox::draw(Painter &p, const PaintContext &context) const {
	p.translate(0, st::msgServiceGiftBoxTopSkip);

	PainterHighQualityEnabler hq(p);
	p.setPen(Qt::NoPen);
	p.setBrush(context.st->msgServiceBg());

	const auto radius = st::msgServiceGiftBoxRadius;
	if (_parent->data()->inlineReplyKeyboard()) {
		const auto r = Rect(_innerSize);
		const auto half = r.height() / 2;
		p.setClipRect(r - QMargins(0, 0, 0, half));
		p.drawRoundedRect(r, radius, radius);
		p.setClipRect(r - QMargins(0, r.height() - half, 0, 0));
		const auto small = Ui::BubbleRadiusSmall();
		p.drawRoundedRect(r, small, small);
		p.setClipping(false);
	} else {
		p.drawRoundedRect(Rect(_innerSize), radius, radius);
	}

	if (_button.stars) {
		const auto &c = context.st->msgServiceFg()->c;
		if ((*_button.lastFg) != c) {
			_button.lastFg->setRgb(c.red(), c.green(), c.blue());
			const auto padding = _button.size.height() / 2;
			_button.stars->setColorOverride(QGradientStops{
				{ 0., anim::with_alpha(c, .3) },
				{ 1., c },
			});
			_button.stars->setCenter(
				Rect(_button.size) - QMargins(padding, 0, padding, 0));
		}
	}

	const auto content = contentRect();
	auto top = content.top() + content.height();
	{
		p.setPen(context.st->msgServiceFg());
		const auto &padding = st::msgServiceGiftBoxTitlePadding;
		top += padding.top();
		if (!_title.isEmpty()) {
			_title.draw(p, {
				.position = QPoint(st::msgPadding.left(), top),
				.availableWidth = _maxWidth,
				.align = style::al_top,
				.palette = &context.st->serviceTextPalette(),
				.spoiler = Ui::Text::DefaultSpoilerCache(),
				.now = context.now,
				.pausedEmoji = context.paused || On(PowerSaving::kEmojiChat),
				.pausedSpoiler = context.paused || On(PowerSaving::kChatSpoiler),
			});
			top += _title.countHeight(_maxWidth) + padding.bottom();
		}
		if (!_author.isEmpty()) {
			auto hq = PainterHighQualityEnabler(p);
			p.setPen(Qt::NoPen);
			p.setBrush(context.st->msgServiceBg());
			const auto use = std::min(_maxWidth, _author.maxWidth())
				+ st::giftBoxReleasedByMargin.left()
				+ st::giftBoxReleasedByMargin.right();
			const auto left = st::msgPadding.left() + (_maxWidth - use) / 2;
			const auto height = st::giftBoxReleasedByMargin.top()
				+ st::uniqueGiftReleasedBy.style.font->height
				+ st::giftBoxReleasedByMargin.bottom();
			const auto radius = height / 2.;
			p.drawRoundedRect(left, top, use, height, radius, radius);

			auto fg = context.st->msgServiceFg()->c;
			fg.setAlphaF(0.65 * fg.alphaF());
			p.setPen(fg);
			_author.draw(p, {
				.position = QPoint(
					left + st::giftBoxReleasedByMargin.left(),
					top + st::giftBoxReleasedByMargin.top()),
				.availableWidth = (use
					- st::giftBoxReleasedByMargin.left()
					- st::giftBoxReleasedByMargin.right()),
				.palette = &context.st->serviceTextPalette(),
				.elisionLines = 1,
			});
			p.setPen(context.st->msgServiceFg());

			top += height + st::msgServiceGiftBoxTitlePadding.bottom();
		}
		_parent->prepareCustomEmojiPaint(p, context, _subtitle);
		_subtitle.draw(p, {
			.position = QPoint(st::msgPadding.left(), top),
			.availableWidth = _maxWidth,
			.align = style::al_top,
			.palette = &context.st->serviceTextPalette(),
			.spoiler = Ui::Text::DefaultSpoilerCache(),
			.now = context.now,
			.pausedEmoji = context.paused || On(PowerSaving::kEmojiChat),
			.pausedSpoiler = context.paused || On(PowerSaving::kChatSpoiler),
		});
		top += _subtitle.countHeight(_maxWidth) + padding.bottom();
	}

	if (!_button.empty()) {
		const auto position = buttonRect().topLeft();
		p.translate(position);

		p.setPen(Qt::NoPen);
		p.setBrush(context.st->msgServiceBg()); // ?
		if (const auto stars = _button.stars.get()) {
			stars->setPaused(context.paused);
		}
		_button.drawBg(p);
		p.setPen(context.st->msgServiceFg());
		if (_button.ripple) {
			const auto opacity = p.opacity();
			p.setOpacity(st::historyPollRippleOpacity);
			_button.ripple->paint(
				p,
				0,
				0,
				width(),
				&context.messageStyle()->msgWaveformInactive->c);
			p.setOpacity(opacity);
		}
		_button.text.draw(
			p,
			0,
			(_button.size.height() - _button.text.minHeight()) / 2,
			_button.size.width(),
			style::al_top);

		p.translate(-position);
	}

	_content->draw(p, context, content);

	if (const auto tag = _content->cornerTag(context); !tag.isNull()) {
		const auto width = tag.width() / tag.devicePixelRatio();
		p.drawImage(_innerSize.width() - width, 0, tag);
	}

	p.translate(0, -st::msgServiceGiftBoxTopSkip);
}

TextState ServiceBox::textState(QPoint point, StateRequest request) const {
	auto result = TextState(_parent);
	point.setY(point.y() - st::msgServiceGiftBoxTopSkip);
	const auto content = contentRect();
	const auto lookupSubtitleLink = [&] {
		auto top = content.top() + content.height();
		const auto &padding = st::msgServiceGiftBoxTitlePadding;
		top += padding.top();
		if (!_title.isEmpty()) {
			top += _title.countHeight(_maxWidth) + padding.bottom();
		}
		if (!_author.isEmpty()) {
			const auto use = std::min(_maxWidth, _author.maxWidth())
				+ st::giftBoxReleasedByMargin.left()
				+ st::giftBoxReleasedByMargin.right();
			const auto left = st::msgPadding.left() + (_maxWidth - use) / 2;
			const auto height = st::giftBoxReleasedByMargin.top()
				+ st::defaultTextStyle.font->height
				+ st::giftBoxReleasedByMargin.bottom();
			if (point.x() >= left
				&& point.y() >= top
				&& point.x() < left + use
				&& point.y() < top + height) {
				result.link = _content->authorLink();
			}
			top += height + st::msgServiceGiftBoxTitlePadding.bottom();
		}

		auto subtitleRequest = request.forText();
		subtitleRequest.align = style::al_top;
		const auto state = _subtitle.getState(
			point - QPoint(st::msgPadding.left(), top),
			_maxWidth,
			subtitleRequest);
		if (state.link) {
			result.link = state.link;
		}
	};
	if (_button.empty()) {
		if (!_button.link) {
			lookupSubtitleLink();
		} else if (QRect(QPoint(), _innerSize).contains(point)) {
			result.link = _button.link;
		}
	} else {
		const auto rect = buttonRect();
		if (rect.contains(point)) {
			result.link = _button.link;
			_button.lastPoint = point - rect.topLeft();
		} else if (content.contains(point)) {
			if (!_contentLink) {
				_contentLink = _content->createViewLink();
			}
			result.link = _contentLink;
		} else {
			lookupSubtitleLink();
		}
	}
	return result;
}

bool ServiceBox::toggleSelectionByHandlerClick(
		const ClickHandlerPtr &p) const {
	return false;
}

bool ServiceBox::dragItemByHandler(const ClickHandlerPtr &p) const {
	return false;
}

void ServiceBox::clickHandlerPressedChanged(
		const ClickHandlerPtr &handler,
		bool pressed) {
	if (!handler) {
		return;
	}

	if (handler == _button.link) {
		_button.toggleRipple(pressed);
	}
}

void ServiceBox::stickerClearLoopPlayed() {
	_content->stickerClearLoopPlayed();
}

std::unique_ptr<StickerPlayer> ServiceBox::stickerTakePlayer(
		not_null<DocumentData*> data,
		const Lottie::ColorReplacements *replacements) {
	return _content->stickerTakePlayer(data, replacements);
}

bool ServiceBox::needsBubble() const {
	return false;
}

bool ServiceBox::customInfoLayout() const {
	return false;
}

void ServiceBox::hideSpoilers() {
	_subtitle.setSpoilerRevealed(false, anim::type::instant);
}

bool ServiceBox::hasHeavyPart() const {
	return _content->hasHeavyPart();
}

void ServiceBox::unloadHeavyPart() {
	_content->unloadHeavyPart();
}

QRect ServiceBox::buttonRect() const {
	const auto &padding = st::msgServiceGiftBoxButtonMargins;
	const auto position = QPoint(
		(width() - _button.size.width()) / 2,
		height() - padding.bottom() - _button.size.height());
	return QRect(position, _button.size);
}

QRect ServiceBox::contentRect() const {
	const auto size = _content->size();
	const auto top = _content->top();
	return QRect(QPoint((width() - size.width()) / 2, top), size);
}

void ServiceBox::Button::toggleRipple(bool pressed) {
	if (empty()) {
		return;
	} else if (pressed) {
		const auto linkWidth = size.width();
		const auto linkHeight = size.height();
		if (!ripple) {
			const auto drawMask = [&](QPainter &p) { drawBg(p); };
			auto mask = Ui::RippleAnimation::MaskByDrawer(
				QSize(linkWidth, linkHeight),
				false,
				drawMask);
			ripple = std::make_unique<Ui::RippleAnimation>(
				st::defaultRippleAnimation,
				std::move(mask),
				repaint);
		}
		ripple->add(lastPoint);
	} else if (ripple) {
		ripple->lastStop();
	}
}

bool ServiceBox::Button::empty() const {
	return text.isEmpty();
}

void ServiceBox::Button::drawBg(QPainter &p) const {
	const auto radius = size.height() / 2.;
	const auto r = Rect(size);
	p.drawRoundedRect(r, radius, radius);
	if (stars) {
		auto clipPath = QPainterPath();
		clipPath.addRoundedRect(r, radius, radius);
		p.setClipPath(clipPath);
		stars->paint(p);
		p.setClipping(false);
	}
}

} // namespace HistoryView
