/**
 * Copyright 2013-2023 Software Radio Systems Limited
 *
 * This file is part of srsRAN.
 *
 * srsRAN is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of
 * the License, or (at your option) any later version.
 *
 * srsRAN is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * A copy of the GNU Affero General Public License can be found in
 * the LICENSE file in the top-level directory of this distribution
 * and at http://www.gnu.org/licenses/.
 *
 */

/*!
 * \file base_graph.c
 * \brief Definition of the two LDPC base graphs employed in the 5G NR
 * standard.
 * \author David Gregoratti
 * \date 2020
 *
 * Look-up tables defining the 51 possible Lifting Sizes (grouped into 8 sets)
 * and the 16 possible Base Graphs (2 main BG indices and 8 variations each
 * according to the Lifting Size set index).
 *
 * Definitions of the functions needed to access these look-up tables.
 *
 * \copyright Software Radio Systems Limited
 *
 */

#include <stdint.h>

#include "srsran/phy/fec/ldpc/base_graph.h"
#include "srsran/phy/utils/debug.h"

/*!
 * \brief Lifting size look-up table.
 *
 * Possible lifting sizes are assigned a lifting-size index (from 0 to 7), the
 * others are marked as VOID_LIFTSIZE.
 */
const uint8_t LSindex[385] = {VOID_LIFTSIZE, // 0
                              VOID_LIFTSIZE, // 1
                              0,             // 2
                              1,             // 3
                              0,             // 4
                              2,             // 5
                              1,             // 6
                              3,             // 7
                              0,             // 8
                              4,             // 9
                              2,             // 10
                              5,             // 11
                              1,             // 12
                              6,             // 13
                              3,             // 14
                              7,             // 15
                              0,             // 16
                              VOID_LIFTSIZE, // 17
                              4,             // 18
                              VOID_LIFTSIZE, // 19
                              2,             // 20
                              VOID_LIFTSIZE, // 21
                              5,             // 22
                              VOID_LIFTSIZE, // 23
                              1,             // 24
                              VOID_LIFTSIZE, // 25
                              6,             // 26
                              VOID_LIFTSIZE, // 27
                              3,             // 28
                              VOID_LIFTSIZE, // 29
                              7,             // 30
                              VOID_LIFTSIZE, // 31
                              0,             // 32
                              VOID_LIFTSIZE, // 33
                              VOID_LIFTSIZE, // 34
                              VOID_LIFTSIZE, // 35
                              4,             // 36
                              VOID_LIFTSIZE, // 37
                              VOID_LIFTSIZE, // 38
                              VOID_LIFTSIZE, // 39
                              2,             // 40
                              VOID_LIFTSIZE, // 41
                              VOID_LIFTSIZE, // 42
                              VOID_LIFTSIZE, // 43
                              5,             // 44
                              VOID_LIFTSIZE, // 45
                              VOID_LIFTSIZE, // 46
                              VOID_LIFTSIZE, // 47
                              1,             // 48
                              VOID_LIFTSIZE, // 49
                              VOID_LIFTSIZE, // 50
                              VOID_LIFTSIZE, // 51
                              6,             // 52
                              VOID_LIFTSIZE, // 53
                              VOID_LIFTSIZE, // 54
                              VOID_LIFTSIZE, // 55
                              3,             // 56
                              VOID_LIFTSIZE, // 57
                              VOID_LIFTSIZE, // 58
                              VOID_LIFTSIZE, // 59
                              7,             // 60
                              VOID_LIFTSIZE, // 61
                              VOID_LIFTSIZE, // 62
                              VOID_LIFTSIZE, // 63
                              0,             // 64
                              VOID_LIFTSIZE, // 65
                              VOID_LIFTSIZE, // 66
                              VOID_LIFTSIZE, // 67
                              VOID_LIFTSIZE, // 68
                              VOID_LIFTSIZE, // 69
                              VOID_LIFTSIZE, // 70
                              VOID_LIFTSIZE, // 71
                              4,             // 72
                              VOID_LIFTSIZE, // 73
                              VOID_LIFTSIZE, // 74
                              VOID_LIFTSIZE, // 75
                              VOID_LIFTSIZE, // 76
                              VOID_LIFTSIZE, // 77
                              VOID_LIFTSIZE, // 78
                              VOID_LIFTSIZE, // 79
                              2,             // 80
                              VOID_LIFTSIZE, // 81
                              VOID_LIFTSIZE, // 82
                              VOID_LIFTSIZE, // 83
                              VOID_LIFTSIZE, // 84
                              VOID_LIFTSIZE, // 85
                              VOID_LIFTSIZE, // 86
                              VOID_LIFTSIZE, // 87
                              5,             // 88
                              VOID_LIFTSIZE, // 89
                              VOID_LIFTSIZE, // 90
                              VOID_LIFTSIZE, // 91
                              VOID_LIFTSIZE, // 92
                              VOID_LIFTSIZE, // 93
                              VOID_LIFTSIZE, // 94
                              VOID_LIFTSIZE, // 95
                              1,             // 96
                              VOID_LIFTSIZE, // 97
                              VOID_LIFTSIZE, // 98
                              VOID_LIFTSIZE, // 99
                              VOID_LIFTSIZE, // 100
                              VOID_LIFTSIZE, // 101
                              VOID_LIFTSIZE, // 102
                              VOID_LIFTSIZE, // 103
                              6,             // 104
                              VOID_LIFTSIZE, // 105
                              VOID_LIFTSIZE, // 106
                              VOID_LIFTSIZE, // 107
                              VOID_LIFTSIZE, // 108
                              VOID_LIFTSIZE, // 109
                              VOID_LIFTSIZE, // 110
                              VOID_LIFTSIZE, // 111
                              3,             // 112
                              VOID_LIFTSIZE, // 113
                              VOID_LIFTSIZE, // 114
                              VOID_LIFTSIZE, // 115
                              VOID_LIFTSIZE, // 116
                              VOID_LIFTSIZE, // 117
                              VOID_LIFTSIZE, // 118
                              VOID_LIFTSIZE, // 119
                              7,             // 120
                              VOID_LIFTSIZE, // 121
                              VOID_LIFTSIZE, // 122
                              VOID_LIFTSIZE, // 123
                              VOID_LIFTSIZE, // 124
                              VOID_LIFTSIZE, // 125
                              VOID_LIFTSIZE, // 126
                              VOID_LIFTSIZE, // 127
                              0,             // 128
                              VOID_LIFTSIZE, // 129
                              VOID_LIFTSIZE, // 130
                              VOID_LIFTSIZE, // 131
                              VOID_LIFTSIZE, // 132
                              VOID_LIFTSIZE, // 133
                              VOID_LIFTSIZE, // 134
                              VOID_LIFTSIZE, // 135
                              VOID_LIFTSIZE, // 136
                              VOID_LIFTSIZE, // 137
                              VOID_LIFTSIZE, // 138
                              VOID_LIFTSIZE, // 139
                              VOID_LIFTSIZE, // 140
                              VOID_LIFTSIZE, // 141
                              VOID_LIFTSIZE, // 142
                              VOID_LIFTSIZE, // 143
                              4,             // 144
                              VOID_LIFTSIZE, // 145
                              VOID_LIFTSIZE, // 146
                              VOID_LIFTSIZE, // 147
                              VOID_LIFTSIZE, // 148
                              VOID_LIFTSIZE, // 149
                              VOID_LIFTSIZE, // 150
                              VOID_LIFTSIZE, // 151
                              VOID_LIFTSIZE, // 152
                              VOID_LIFTSIZE, // 153
                              VOID_LIFTSIZE, // 154
                              VOID_LIFTSIZE, // 155
                              VOID_LIFTSIZE, // 156
                              VOID_LIFTSIZE, // 157
                              VOID_LIFTSIZE, // 158
                              VOID_LIFTSIZE, // 159
                              2,             // 160
                              VOID_LIFTSIZE, // 161
                              VOID_LIFTSIZE, // 162
                              VOID_LIFTSIZE, // 163
                              VOID_LIFTSIZE, // 164
                              VOID_LIFTSIZE, // 165
                              VOID_LIFTSIZE, // 166
                              VOID_LIFTSIZE, // 167
                              VOID_LIFTSIZE, // 168
                              VOID_LIFTSIZE, // 169
                              VOID_LIFTSIZE, // 170
                              VOID_LIFTSIZE, // 171
                              VOID_LIFTSIZE, // 172
                              VOID_LIFTSIZE, // 173
                              VOID_LIFTSIZE, // 174
                              VOID_LIFTSIZE, // 175
                              5,             // 176
                              VOID_LIFTSIZE, // 177
                              VOID_LIFTSIZE, // 178
                              VOID_LIFTSIZE, // 179
                              VOID_LIFTSIZE, // 180
                              VOID_LIFTSIZE, // 181
                              VOID_LIFTSIZE, // 182
                              VOID_LIFTSIZE, // 183
                              VOID_LIFTSIZE, // 184
                              VOID_LIFTSIZE, // 185
                              VOID_LIFTSIZE, // 186
                              VOID_LIFTSIZE, // 187
                              VOID_LIFTSIZE, // 188
                              VOID_LIFTSIZE, // 189
                              VOID_LIFTSIZE, // 190
                              VOID_LIFTSIZE, // 191
                              1,             // 192
                              VOID_LIFTSIZE, // 193
                              VOID_LIFTSIZE, // 194
                              VOID_LIFTSIZE, // 195
                              VOID_LIFTSIZE, // 196
                              VOID_LIFTSIZE, // 197
                              VOID_LIFTSIZE, // 198
                              VOID_LIFTSIZE, // 199
                              VOID_LIFTSIZE, // 200
                              VOID_LIFTSIZE, // 201
                              VOID_LIFTSIZE, // 202
                              VOID_LIFTSIZE, // 203
                              VOID_LIFTSIZE, // 204
                              VOID_LIFTSIZE, // 205
                              VOID_LIFTSIZE, // 206
                              VOID_LIFTSIZE, // 207
                              6,             // 208
                              VOID_LIFTSIZE, // 209
                              VOID_LIFTSIZE, // 210
                              VOID_LIFTSIZE, // 211
                              VOID_LIFTSIZE, // 212
                              VOID_LIFTSIZE, // 213
                              VOID_LIFTSIZE, // 214
                              VOID_LIFTSIZE, // 215
                              VOID_LIFTSIZE, // 216
                              VOID_LIFTSIZE, // 217
                              VOID_LIFTSIZE, // 218
                              VOID_LIFTSIZE, // 219
                              VOID_LIFTSIZE, // 220
                              VOID_LIFTSIZE, // 221
                              VOID_LIFTSIZE, // 222
                              VOID_LIFTSIZE, // 223
                              3,             // 224
                              VOID_LIFTSIZE, // 225
                              VOID_LIFTSIZE, // 226
                              VOID_LIFTSIZE, // 227
                              VOID_LIFTSIZE, // 228
                              VOID_LIFTSIZE, // 229
                              VOID_LIFTSIZE, // 230
                              VOID_LIFTSIZE, // 231
                              VOID_LIFTSIZE, // 232
                              VOID_LIFTSIZE, // 233
                              VOID_LIFTSIZE, // 234
                              VOID_LIFTSIZE, // 235
                              VOID_LIFTSIZE, // 236
                              VOID_LIFTSIZE, // 237
                              VOID_LIFTSIZE, // 238
                              VOID_LIFTSIZE, // 239
                              7,             // 240
                              VOID_LIFTSIZE, // 241
                              VOID_LIFTSIZE, // 242
                              VOID_LIFTSIZE, // 243
                              VOID_LIFTSIZE, // 244
                              VOID_LIFTSIZE, // 245
                              VOID_LIFTSIZE, // 246
                              VOID_LIFTSIZE, // 247
                              VOID_LIFTSIZE, // 248
                              VOID_LIFTSIZE, // 249
                              VOID_LIFTSIZE, // 250
                              VOID_LIFTSIZE, // 251
                              VOID_LIFTSIZE, // 252
                              VOID_LIFTSIZE, // 253
                              VOID_LIFTSIZE, // 254
                              VOID_LIFTSIZE, // 255
                              0,             // 256
                              VOID_LIFTSIZE, // 257
                              VOID_LIFTSIZE, // 258
                              VOID_LIFTSIZE, // 259
                              VOID_LIFTSIZE, // 260
                              VOID_LIFTSIZE, // 261
                              VOID_LIFTSIZE, // 262
                              VOID_LIFTSIZE, // 263
                              VOID_LIFTSIZE, // 264
                              VOID_LIFTSIZE, // 265
                              VOID_LIFTSIZE, // 266
                              VOID_LIFTSIZE, // 267
                              VOID_LIFTSIZE, // 268
                              VOID_LIFTSIZE, // 269
                              VOID_LIFTSIZE, // 270
                              VOID_LIFTSIZE, // 271
                              VOID_LIFTSIZE, // 272
                              VOID_LIFTSIZE, // 273
                              VOID_LIFTSIZE, // 274
                              VOID_LIFTSIZE, // 275
                              VOID_LIFTSIZE, // 276
                              VOID_LIFTSIZE, // 277
                              VOID_LIFTSIZE, // 278
                              VOID_LIFTSIZE, // 279
                              VOID_LIFTSIZE, // 280
                              VOID_LIFTSIZE, // 281
                              VOID_LIFTSIZE, // 282
                              VOID_LIFTSIZE, // 283
                              VOID_LIFTSIZE, // 284
                              VOID_LIFTSIZE, // 285
                              VOID_LIFTSIZE, // 286
                              VOID_LIFTSIZE, // 287
                              4,             // 288
                              VOID_LIFTSIZE, // 289
                              VOID_LIFTSIZE, // 290
                              VOID_LIFTSIZE, // 291
                              VOID_LIFTSIZE, // 292
                              VOID_LIFTSIZE, // 293
                              VOID_LIFTSIZE, // 294
                              VOID_LIFTSIZE, // 295
                              VOID_LIFTSIZE, // 296
                              VOID_LIFTSIZE, // 297
                              VOID_LIFTSIZE, // 298
                              VOID_LIFTSIZE, // 299
                              VOID_LIFTSIZE, // 300
                              VOID_LIFTSIZE, // 301
                              VOID_LIFTSIZE, // 302
                              VOID_LIFTSIZE, // 303
                              VOID_LIFTSIZE, // 304
                              VOID_LIFTSIZE, // 305
                              VOID_LIFTSIZE, // 306
                              VOID_LIFTSIZE, // 307
                              VOID_LIFTSIZE, // 308
                              VOID_LIFTSIZE, // 309
                              VOID_LIFTSIZE, // 310
                              VOID_LIFTSIZE, // 311
                              VOID_LIFTSIZE, // 312
                              VOID_LIFTSIZE, // 313
                              VOID_LIFTSIZE, // 314
                              VOID_LIFTSIZE, // 315
                              VOID_LIFTSIZE, // 316
                              VOID_LIFTSIZE, // 317
                              VOID_LIFTSIZE, // 318
                              VOID_LIFTSIZE, // 319
                              2,             // 320
                              VOID_LIFTSIZE, // 321
                              VOID_LIFTSIZE, // 322
                              VOID_LIFTSIZE, // 323
                              VOID_LIFTSIZE, // 324
                              VOID_LIFTSIZE, // 325
                              VOID_LIFTSIZE, // 326
                              VOID_LIFTSIZE, // 327
                              VOID_LIFTSIZE, // 328
                              VOID_LIFTSIZE, // 329
                              VOID_LIFTSIZE, // 330
                              VOID_LIFTSIZE, // 331
                              VOID_LIFTSIZE, // 332
                              VOID_LIFTSIZE, // 333
                              VOID_LIFTSIZE, // 334
                              VOID_LIFTSIZE, // 335
                              VOID_LIFTSIZE, // 336
                              VOID_LIFTSIZE, // 337
                              VOID_LIFTSIZE, // 338
                              VOID_LIFTSIZE, // 339
                              VOID_LIFTSIZE, // 340
                              VOID_LIFTSIZE, // 341
                              VOID_LIFTSIZE, // 342
                              VOID_LIFTSIZE, // 343
                              VOID_LIFTSIZE, // 344
                              VOID_LIFTSIZE, // 345
                              VOID_LIFTSIZE, // 346
                              VOID_LIFTSIZE, // 347
                              VOID_LIFTSIZE, // 348
                              VOID_LIFTSIZE, // 349
                              VOID_LIFTSIZE, // 350
                              VOID_LIFTSIZE, // 351
                              5,             // 352
                              VOID_LIFTSIZE, // 353
                              VOID_LIFTSIZE, // 354
                              VOID_LIFTSIZE, // 355
                              VOID_LIFTSIZE, // 356
                              VOID_LIFTSIZE, // 357
                              VOID_LIFTSIZE, // 358
                              VOID_LIFTSIZE, // 359
                              VOID_LIFTSIZE, // 360
                              VOID_LIFTSIZE, // 361
                              VOID_LIFTSIZE, // 362
                              VOID_LIFTSIZE, // 363
                              VOID_LIFTSIZE, // 364
                              VOID_LIFTSIZE, // 365
                              VOID_LIFTSIZE, // 366
                              VOID_LIFTSIZE, // 367
                              VOID_LIFTSIZE, // 368
                              VOID_LIFTSIZE, // 369
                              VOID_LIFTSIZE, // 370
                              VOID_LIFTSIZE, // 371
                              VOID_LIFTSIZE, // 372
                              VOID_LIFTSIZE, // 373
                              VOID_LIFTSIZE, // 374
                              VOID_LIFTSIZE, // 375
                              VOID_LIFTSIZE, // 376
                              VOID_LIFTSIZE, // 377
                              VOID_LIFTSIZE, // 378
                              VOID_LIFTSIZE, // 379
                              VOID_LIFTSIZE, // 380
                              VOID_LIFTSIZE, // 381
                              VOID_LIFTSIZE, // 382
                              VOID_LIFTSIZE, // 383
                              1};            // 384

/*!
 * \brief Look-up table for Base Graph BG1.
 *
 * BG1 has sizes 46x68. There are 8 possible variations according to the lifting
 * size index. Each entry corresponds to the cyclic rotation applied to the
 * (check node)--(variable node) connection when lifting the graph. Entries
 * marked as NO_CNCT mean that there is no connection between the corresponding
 * check node and variable node (in both the base graph and the lifted one).
 */
static const uint16_t BG1_matrices[NOF_LIFTSIZE][BG1M][BG1Nfull] = {
    // BG1 - lifting size index 0
    {{250,     69,      226,     159,     NO_CNCT, 100,     10,      NO_CNCT, NO_CNCT, 59,      229,     110,
      191,     9,       NO_CNCT, 195,     23,      NO_CNCT, 190,     35,      239,     31,      1,       0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {2,       NO_CNCT, 239,     117,     124,     71,      NO_CNCT, 222,     104,     173,     NO_CNCT, 220,
      102,     NO_CNCT, 109,     132,     142,     155,     NO_CNCT, 255,     NO_CNCT, 28,      0,       0,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {106,     111,     185,     NO_CNCT, 63,      117,     93,      229,     177,     95,      39,      NO_CNCT,
      NO_CNCT, 142,     225,     225,     NO_CNCT, 245,     205,     251,     117,     NO_CNCT, NO_CNCT, NO_CNCT,
      0,       0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {121,     89,      NO_CNCT, 84,      20,      NO_CNCT, 150,     131,     243,     NO_CNCT, 136,     86,
      246,     219,     211,     NO_CNCT, 240,     76,      244,     NO_CNCT, 144,     12,      1,       NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {157,     102,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {205,     236,     NO_CNCT, 194,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      231,     NO_CNCT, NO_CNCT, NO_CNCT, 28,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 123,     115,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {183,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 22,      NO_CNCT, NO_CNCT, NO_CNCT, 28,      67,
      NO_CNCT, 244,     NO_CNCT, NO_CNCT, NO_CNCT, 11,      157,     NO_CNCT, 211,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {220,     44,      NO_CNCT, NO_CNCT, 159,     NO_CNCT, NO_CNCT, 31,      167,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 104,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {112,     4,       NO_CNCT, 7,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      211,     NO_CNCT, NO_CNCT, NO_CNCT, 102,     NO_CNCT, NO_CNCT, 164,     NO_CNCT, 109,     241,     NO_CNCT,
      90,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {103,     182,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 109,     21,
      NO_CNCT, 142,     NO_CNCT, NO_CNCT, NO_CNCT, 14,      61,      NO_CNCT, 216,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 98,      149,     NO_CNCT, 167,     NO_CNCT, NO_CNCT, 160,     49,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 58,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {77,      41,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      83,      NO_CNCT, NO_CNCT, NO_CNCT, 182,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 78,      252,     22,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {160,     42,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 21,      32,
      NO_CNCT, 234,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 7,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {177,     NO_CNCT, NO_CNCT, 248,     NO_CNCT, NO_CNCT, NO_CNCT, 151,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 185,     NO_CNCT, NO_CNCT, 62,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {206,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      55,      NO_CNCT, NO_CNCT, 206,     127,     16,      NO_CNCT, NO_CNCT, NO_CNCT, 229,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {40,      96,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 65,      NO_CNCT,
      NO_CNCT, 63,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 75,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 179,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 64,      NO_CNCT, 49,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 49,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 51,      NO_CNCT, 154,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {7,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 164,     NO_CNCT, 59,      1,       NO_CNCT, NO_CNCT, NO_CNCT, 144,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 42,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      233,     8,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 155,     147,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {60,      73,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 72,      127,     NO_CNCT, 224,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {151,     NO_CNCT, NO_CNCT, 186,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 217,     NO_CNCT, 47,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 160,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 249,     NO_CNCT, NO_CNCT, NO_CNCT, 121,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 109,     NO_CNCT, NO_CNCT, NO_CNCT, 131,     171,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {64,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      142,     188,     NO_CNCT, NO_CNCT, NO_CNCT, 158,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 156,     147,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 170,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 152,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {112,     NO_CNCT, NO_CNCT, 86,      236,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 116,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 222,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 23,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 136,     116,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 182,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {195,     NO_CNCT, 243,     NO_CNCT, 215,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 61,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 25,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 104,     NO_CNCT, 194,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {128,     NO_CNCT, NO_CNCT, NO_CNCT, 165,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 181,     NO_CNCT, 63,      NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 86,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 236,     NO_CNCT, NO_CNCT, NO_CNCT, 84,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 6,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {216,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 73,      NO_CNCT,
      NO_CNCT, 120,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      9,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 95,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 177,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 172,     NO_CNCT,
      NO_CNCT, 61,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {221,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      112,     NO_CNCT, 199,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      121,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 2,       187,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 41,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 211,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {127,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 167,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 164,     NO_CNCT, 159,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 161,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 197,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      207,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 103,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {37,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 105,     51,      NO_CNCT, NO_CNCT, 120,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 198,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 220,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 122,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {167,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 151,     157,     NO_CNCT,
      163,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 173,     NO_CNCT, 139,     NO_CNCT, NO_CNCT, NO_CNCT, 149,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {157,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 137,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 149,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 167,     NO_CNCT, 173,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 139,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 151,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {149,     NO_CNCT, NO_CNCT, NO_CNCT, 157,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      137,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 151,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 163,     NO_CNCT, 173,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 139,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT},
     {139,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 157,     NO_CNCT, 163,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 173,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT},
     {NO_CNCT, 149,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 151,     NO_CNCT, NO_CNCT, NO_CNCT, 167,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0}},

    // BG1 - lifting size index 1
    {{307,     19,      50,      369,     NO_CNCT, 181,     216,     NO_CNCT, NO_CNCT, 317,     288,     109,
      17,      357,     NO_CNCT, 215,     106,     NO_CNCT, 242,     180,     330,     346,     1,       0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {76,      NO_CNCT, 76,      73,      288,     144,     NO_CNCT, 331,     331,     178,     NO_CNCT, 295,
      342,     NO_CNCT, 217,     99,      354,     114,     NO_CNCT, 331,     NO_CNCT, 112,     0,       0,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {205,     250,     328,     NO_CNCT, 332,     256,     161,     267,     160,     63,      129,     NO_CNCT,
      NO_CNCT, 200,     88,      53,      NO_CNCT, 131,     240,     205,     13,      NO_CNCT, NO_CNCT, NO_CNCT,
      0,       0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {276,     87,      NO_CNCT, 0,       275,     NO_CNCT, 199,     153,     56,      NO_CNCT, 132,     305,
      231,     341,     212,     NO_CNCT, 304,     300,     271,     NO_CNCT, 39,      357,     1,       NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {332,     181,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {195,     14,      NO_CNCT, 115,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      166,     NO_CNCT, NO_CNCT, NO_CNCT, 241,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 51,      157,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {278,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 257,     NO_CNCT, NO_CNCT, NO_CNCT, 1,       351,
      NO_CNCT, 92,      NO_CNCT, NO_CNCT, NO_CNCT, 253,     18,      NO_CNCT, 225,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {9,       62,      NO_CNCT, NO_CNCT, 316,     NO_CNCT, NO_CNCT, 333,     290,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 114,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {307,     179,     NO_CNCT, 165,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      18,      NO_CNCT, NO_CNCT, NO_CNCT, 39,      NO_CNCT, NO_CNCT, 224,     NO_CNCT, 368,     67,      NO_CNCT,
      170,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {366,     232,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 321,     133,
      NO_CNCT, 57,      NO_CNCT, NO_CNCT, NO_CNCT, 303,     63,      NO_CNCT, 82,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 101,     339,     NO_CNCT, 274,     NO_CNCT, NO_CNCT, 111,     383,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 354,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {48,      102,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      8,       NO_CNCT, NO_CNCT, NO_CNCT, 47,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 188,     334,     115,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {77,      186,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 174,     232,
      NO_CNCT, 50,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 74,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {313,     NO_CNCT, NO_CNCT, 177,     NO_CNCT, NO_CNCT, NO_CNCT, 266,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 115,     NO_CNCT, NO_CNCT, 370,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {142,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      248,     NO_CNCT, NO_CNCT, 137,     89,      347,     NO_CNCT, NO_CNCT, NO_CNCT, 12,      NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {241,     2,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 210,     NO_CNCT,
      NO_CNCT, 318,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 55,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 269,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 13,      NO_CNCT, 338,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 57,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 289,     NO_CNCT, 57,      NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {260,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 303,     NO_CNCT, 81,      358,     NO_CNCT, NO_CNCT, NO_CNCT, 375,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 130,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      163,     280,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 132,     4,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {145,     213,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 344,     242,     NO_CNCT, 197,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {187,     NO_CNCT, NO_CNCT, 206,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 264,     NO_CNCT, 341,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 59,      NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 205,     NO_CNCT, NO_CNCT, NO_CNCT, 102,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 328,     NO_CNCT, NO_CNCT, NO_CNCT, 213,     97,      NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {30,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      11,      233,     NO_CNCT, NO_CNCT, NO_CNCT, 22,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 24,      89,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 61,      NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 27,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {298,     NO_CNCT, NO_CNCT, 158,     235,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 339,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 234,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 72,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 17,      383,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 312,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {71,      NO_CNCT, 81,      NO_CNCT, 76,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 136,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 194,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 194,     NO_CNCT, 101,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {222,     NO_CNCT, NO_CNCT, NO_CNCT, 19,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 244,     NO_CNCT, 274,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 252,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 5,       NO_CNCT, NO_CNCT, NO_CNCT, 147,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 78,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {159,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 229,     NO_CNCT,
      NO_CNCT, 260,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      90,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 100,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 215,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 258,     NO_CNCT,
      NO_CNCT, 256,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {102,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      201,     NO_CNCT, 175,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      287,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 323,     8,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 361,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 105,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {230,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 148,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 202,     NO_CNCT, 312,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 320,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 335,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      2,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 266,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {210,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 313,     297,     NO_CNCT, NO_CNCT, 21,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 269,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 82,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 115,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {185,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 177,     289,     NO_CNCT,
      214,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 258,     NO_CNCT, 93,      NO_CNCT, NO_CNCT, NO_CNCT, 346,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 297,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {175,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 37,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 312,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 52,      NO_CNCT, 314,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 139,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 288,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {113,     NO_CNCT, NO_CNCT, NO_CNCT, 14,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      218,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 113,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 132,     NO_CNCT, 114,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 168,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT},
     {80,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 78,      NO_CNCT, 163,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 274,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT},
     {NO_CNCT, 135,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 149,     NO_CNCT, NO_CNCT, NO_CNCT, 15,      NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0}},

    // BG1 - lifting size index 2
    {{73,      15,      103,     49,      NO_CNCT, 240,     39,      NO_CNCT, NO_CNCT, 15,      162,     215,
      164,     133,     NO_CNCT, 298,     110,     NO_CNCT, 113,     16,      189,     32,      1,       0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {303,     NO_CNCT, 294,     27,      261,     161,     NO_CNCT, 133,     4,       80,      NO_CNCT, 129,
      300,     NO_CNCT, 76,      266,     72,      83,      NO_CNCT, 260,     NO_CNCT, 301,     0,       0,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {68,      7,       80,      NO_CNCT, 280,     38,      227,     202,     200,     71,      106,     NO_CNCT,
      NO_CNCT, 295,     283,     301,     NO_CNCT, 184,     246,     230,     276,     NO_CNCT, NO_CNCT, NO_CNCT,
      0,       0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {220,     208,     NO_CNCT, 30,      197,     NO_CNCT, 61,      175,     79,      NO_CNCT, 281,     303,
      253,     164,     53,      NO_CNCT, 44,      28,      77,      NO_CNCT, 319,     68,      1,       NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {233,     205,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {83,      292,     NO_CNCT, 50,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      318,     NO_CNCT, NO_CNCT, NO_CNCT, 201,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 267,     279,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {289,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 21,      NO_CNCT, NO_CNCT, NO_CNCT, 293,     13,
      NO_CNCT, 232,     NO_CNCT, NO_CNCT, NO_CNCT, 302,     138,     NO_CNCT, 235,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {12,      88,      NO_CNCT, NO_CNCT, 207,     NO_CNCT, NO_CNCT, 50,      25,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 76,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {295,     133,     NO_CNCT, 130,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      231,     NO_CNCT, NO_CNCT, NO_CNCT, 296,     NO_CNCT, NO_CNCT, 110,     NO_CNCT, 269,     245,     NO_CNCT,
      154,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {189,     244,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 36,      286,
      NO_CNCT, 151,     NO_CNCT, NO_CNCT, NO_CNCT, 267,     135,     NO_CNCT, 209,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 14,      80,      NO_CNCT, 211,     NO_CNCT, NO_CNCT, 75,      161,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 311,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {16,      147,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      290,     NO_CNCT, NO_CNCT, NO_CNCT, 289,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 177,     43,      280,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {229,     235,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 169,     48,
      NO_CNCT, 105,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 52,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {39,      NO_CNCT, NO_CNCT, 302,     NO_CNCT, NO_CNCT, NO_CNCT, 303,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 160,     NO_CNCT, NO_CNCT, 37,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {78,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      299,     NO_CNCT, NO_CNCT, 54,      61,      179,     NO_CNCT, NO_CNCT, NO_CNCT, 258,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {229,     290,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 60,      NO_CNCT,
      NO_CNCT, 130,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 184,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 51,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 69,      NO_CNCT, 140,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 45,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 115,     NO_CNCT, 300,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {257,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 147,     NO_CNCT, 128,     51,      NO_CNCT, NO_CNCT, NO_CNCT, 228,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 260,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      294,     291,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 141,     295,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {64,      181,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 101,     270,     NO_CNCT, 41,      NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {301,     NO_CNCT, NO_CNCT, 162,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 40,      NO_CNCT, 130,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 10,      NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 79,      NO_CNCT, NO_CNCT, NO_CNCT, 175,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 132,     NO_CNCT, NO_CNCT, NO_CNCT, 283,     103,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {177,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      20,      55,      NO_CNCT, NO_CNCT, NO_CNCT, 316,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 249,     50,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 133,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 105,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {289,     NO_CNCT, NO_CNCT, 280,     110,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 187,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 281,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 172,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 295,     96,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 46,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {270,     NO_CNCT, 110,     NO_CNCT, 318,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 67,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 210,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 29,      NO_CNCT, 304,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {11,      NO_CNCT, NO_CNCT, NO_CNCT, 293,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 50,      NO_CNCT, 234,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 27,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 308,     NO_CNCT, NO_CNCT, NO_CNCT, 117,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 29,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {91,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 23,      NO_CNCT,
      NO_CNCT, 105,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      135,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 222,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 308,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 66,      NO_CNCT,
      NO_CNCT, 162,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {210,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      22,      NO_CNCT, 271,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      217,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 170,     20,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 140,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 33,      NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {187,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 296,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 5,       NO_CNCT, 44,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 207,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 158,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      55,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 285,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {259,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 179,     178,     NO_CNCT, NO_CNCT, 160,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 298,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 15,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 115,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {151,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 179,     64,      NO_CNCT,
      181,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 102,     NO_CNCT, 77,      NO_CNCT, NO_CNCT, NO_CNCT, 192,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 208,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {32,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 80,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 197,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 154,     NO_CNCT, 47,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 124,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 207,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {226,     NO_CNCT, NO_CNCT, NO_CNCT, 65,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      126,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 228,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 69,      NO_CNCT, 176,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 102,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT},
     {234,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 227,     NO_CNCT, 259,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 260,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT},
     {NO_CNCT, 101,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 228,     NO_CNCT, NO_CNCT, NO_CNCT, 126,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0}},

    // BG1 - lifting size index 3
    {{223,     16,      94,      91,      NO_CNCT, 74,      10,      NO_CNCT, NO_CNCT, 0,       205,     216,
      21,      215,     NO_CNCT, 14,      70,      NO_CNCT, 141,     198,     104,     81,      1,       0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {141,     NO_CNCT, 45,      151,     46,      119,     NO_CNCT, 157,     133,     87,      NO_CNCT, 206,
      93,      NO_CNCT, 79,      9,       118,     194,     NO_CNCT, 31,      NO_CNCT, 187,     0,       0,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {207,     203,     31,      NO_CNCT, 176,     180,     186,     95,      153,     177,     70,      NO_CNCT,
      NO_CNCT, 77,      214,     77,      NO_CNCT, 198,     117,     223,     90,      NO_CNCT, NO_CNCT, NO_CNCT,
      0,       0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {201,     18,      NO_CNCT, 165,     5,       NO_CNCT, 45,      142,     16,      NO_CNCT, 34,      155,
      213,     147,     69,      NO_CNCT, 96,      74,      99,      NO_CNCT, 30,      158,     1,       NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {170,     10,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {164,     59,      NO_CNCT, 86,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      80,      NO_CNCT, NO_CNCT, NO_CNCT, 182,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 130,     153,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {158,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 119,     NO_CNCT, NO_CNCT, NO_CNCT, 113,     21,
      NO_CNCT, 63,      NO_CNCT, NO_CNCT, NO_CNCT, 51,      136,     NO_CNCT, 116,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {17,      76,      NO_CNCT, NO_CNCT, 104,     NO_CNCT, NO_CNCT, 100,     150,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 158,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {33,      95,      NO_CNCT, 4,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      217,     NO_CNCT, NO_CNCT, NO_CNCT, 204,     NO_CNCT, NO_CNCT, 39,      NO_CNCT, 58,      44,      NO_CNCT,
      201,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {9,       37,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 213,     105,
      NO_CNCT, 89,      NO_CNCT, NO_CNCT, NO_CNCT, 185,     109,     NO_CNCT, 218,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 82,      165,     NO_CNCT, 174,     NO_CNCT, NO_CNCT, 19,      194,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 103,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {52,      11,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      2,       NO_CNCT, NO_CNCT, NO_CNCT, 35,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 32,      84,      201,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {142,     175,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 136,     3,
      NO_CNCT, 28,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 182,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {81,      NO_CNCT, NO_CNCT, 56,      NO_CNCT, NO_CNCT, NO_CNCT, 72,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 217,     NO_CNCT, NO_CNCT, 78,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {14,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      175,     NO_CNCT, NO_CNCT, 211,     191,     51,      NO_CNCT, NO_CNCT, NO_CNCT, 43,      NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {90,      120,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 131,     NO_CNCT,
      NO_CNCT, 209,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 209,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 81,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 154,     NO_CNCT, 164,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 43,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 189,     NO_CNCT, 101,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {56,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 110,     NO_CNCT, 200,     63,      NO_CNCT, NO_CNCT, NO_CNCT, 4,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 199,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      110,     200,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 143,     186,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {8,       6,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 103,     198,     NO_CNCT, 8,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {105,     NO_CNCT, NO_CNCT, 210,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 121,     NO_CNCT, 214,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 183,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 192,     NO_CNCT, NO_CNCT, NO_CNCT, 131,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 220,     NO_CNCT, NO_CNCT, NO_CNCT, 50,      106,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {53,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       3,       NO_CNCT, NO_CNCT, NO_CNCT, 148,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 88,      203,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 168,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 122,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {49,      NO_CNCT, NO_CNCT, 157,     64,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 193,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 124,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 1,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 166,     65,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 81,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {107,     NO_CNCT, 176,     NO_CNCT, 212,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 127,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 208,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 141,     NO_CNCT, 174,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {146,     NO_CNCT, NO_CNCT, NO_CNCT, 153,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 217,     NO_CNCT, 114,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 150,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 11,      NO_CNCT, NO_CNCT, NO_CNCT, 53,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 68,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {34,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 130,     NO_CNCT,
      NO_CNCT, 210,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      123,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 175,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 49,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 177,     NO_CNCT,
      NO_CNCT, 128,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {192,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      209,     NO_CNCT, 58,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      30,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 114,     49,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 161,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 137,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {82,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 186,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 68,      NO_CNCT, 150,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 192,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 173,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      26,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 187,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {222,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 157,     0,       NO_CNCT, NO_CNCT, 6,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 81,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 195,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 138,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {123,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 90,      73,      NO_CNCT,
      10,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 12,      NO_CNCT, 77,      NO_CNCT, NO_CNCT, NO_CNCT, 49,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 114,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {67,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 45,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 96,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 23,      NO_CNCT, 215,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 60,      NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 167,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {114,     NO_CNCT, NO_CNCT, NO_CNCT, 91,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      78,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 206,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 22,      NO_CNCT, 134,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 161,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT},
     {84,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 4,       NO_CNCT, 9,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 12,      NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT},
     {NO_CNCT, 184,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 121,     NO_CNCT, NO_CNCT, NO_CNCT, 29,      NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0}},

    // BG1 - lifting size index 4
    {{211,     198,     188,     186,     NO_CNCT, 219,     4,       NO_CNCT, NO_CNCT, 29,      144,     116,
      216,     115,     NO_CNCT, 233,     144,     NO_CNCT, 95,      216,     73,      261,     1,       0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {179,     NO_CNCT, 162,     223,     256,     160,     NO_CNCT, 76,      202,     117,     NO_CNCT, 109,
      15,      NO_CNCT, 72,      152,     158,     147,     NO_CNCT, 156,     NO_CNCT, 119,     0,       0,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {258,     167,     220,     NO_CNCT, 133,     243,     202,     218,     63,      0,       3,       NO_CNCT,
      NO_CNCT, 74,      229,     0,       NO_CNCT, 216,     269,     200,     234,     NO_CNCT, NO_CNCT, NO_CNCT,
      0,       0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {187,     145,     NO_CNCT, 166,     108,     NO_CNCT, 82,      132,     197,     NO_CNCT, 41,      162,
      57,      36,      115,     NO_CNCT, 242,     165,     0,       NO_CNCT, 113,     108,     1,       NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {246,     235,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {261,     181,     NO_CNCT, 72,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      283,     NO_CNCT, NO_CNCT, NO_CNCT, 254,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 79,      144,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {80,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 144,     NO_CNCT, NO_CNCT, NO_CNCT, 169,     90,
      NO_CNCT, 59,      NO_CNCT, NO_CNCT, NO_CNCT, 177,     151,     NO_CNCT, 108,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {169,     189,     NO_CNCT, NO_CNCT, 154,     NO_CNCT, NO_CNCT, 184,     104,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 164,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {54,      0,       NO_CNCT, 252,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      41,      NO_CNCT, NO_CNCT, NO_CNCT, 98,      NO_CNCT, NO_CNCT, 46,      NO_CNCT, 15,      230,     NO_CNCT,
      54,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {162,     159,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 93,      134,
      NO_CNCT, 45,      NO_CNCT, NO_CNCT, NO_CNCT, 132,     76,      NO_CNCT, 209,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 178,     1,       NO_CNCT, 28,      NO_CNCT, NO_CNCT, 267,     234,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 201,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {55,      23,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      274,     NO_CNCT, NO_CNCT, NO_CNCT, 181,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 273,     39,      26,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {225,     162,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 244,     151,
      NO_CNCT, 238,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 243,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {231,     NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, 216,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 47,      NO_CNCT, NO_CNCT, 36,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      186,     NO_CNCT, NO_CNCT, 253,     16,      0,       NO_CNCT, NO_CNCT, NO_CNCT, 79,      NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {170,     0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 183,     NO_CNCT,
      NO_CNCT, 108,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 68,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 64,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 270,     NO_CNCT, 13,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 99,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 54,      NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {153,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 137,     NO_CNCT, 0,       0,       NO_CNCT, NO_CNCT, NO_CNCT, 162,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 161,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      151,     0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 241,     144,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {0,       0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 118,     144,     NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {265,     NO_CNCT, NO_CNCT, 81,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 90,      NO_CNCT, 144,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 228,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 64,      NO_CNCT, NO_CNCT, NO_CNCT, 46,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 266,     NO_CNCT, NO_CNCT, NO_CNCT, 9,       18,      NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {72,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      189,     72,      NO_CNCT, NO_CNCT, NO_CNCT, 257,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 180,     0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 165,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {236,     NO_CNCT, NO_CNCT, 199,     0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 266,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 205,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 183,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {0,       NO_CNCT, 0,       NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 277,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 45,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 36,      NO_CNCT, 72,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {275,     NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 155,     NO_CNCT, 62,      NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 180,     NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 42,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 90,      NO_CNCT,
      NO_CNCT, 252,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      173,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 144,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 144,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 166,     NO_CNCT,
      NO_CNCT, 19,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      211,     NO_CNCT, 36,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      162,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 0,       0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 76,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 18,      NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {197,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 108,     NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 199,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 278,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 205,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {216,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 16,      0,       NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 72,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 144,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {190,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       0,       NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 153,     NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, 165,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 117,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {216,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 144,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 2,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 0,       NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 183,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {27,      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      35,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 52,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 243,     NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 270,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT},
     {18,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 57,      NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT},
     {NO_CNCT, 168,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, 144,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0}},

    // BG1 - lifting size index 5
    {{294,     118,     167,     330,     NO_CNCT, 207,     165,     NO_CNCT, NO_CNCT, 243,     250,     1,
      339,     201,     NO_CNCT, 53,      347,     NO_CNCT, 304,     167,     47,      188,     1,       0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {77,      NO_CNCT, 225,     96,      338,     268,     NO_CNCT, 112,     302,     50,      NO_CNCT, 167,
      253,     NO_CNCT, 334,     242,     257,     133,     NO_CNCT, 9,       NO_CNCT, 302,     0,       0,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {226,     35,      213,     NO_CNCT, 302,     111,     265,     128,     237,     294,     127,     NO_CNCT,
      NO_CNCT, 110,     286,     125,     NO_CNCT, 131,     163,     210,     7,       NO_CNCT, NO_CNCT, NO_CNCT,
      0,       0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {97,      94,      NO_CNCT, 49,      279,     NO_CNCT, 139,     166,     91,      NO_CNCT, 106,     246,
      345,     269,     185,     NO_CNCT, 249,     215,     143,     NO_CNCT, 121,     121,     1,       NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {42,      256,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {219,     130,     NO_CNCT, 251,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      322,     NO_CNCT, NO_CNCT, NO_CNCT, 295,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 258,     283,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {294,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 73,      NO_CNCT, NO_CNCT, NO_CNCT, 330,     99,
      NO_CNCT, 172,     NO_CNCT, NO_CNCT, NO_CNCT, 150,     284,     NO_CNCT, 305,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {3,       103,     NO_CNCT, NO_CNCT, 224,     NO_CNCT, NO_CNCT, 297,     215,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 39,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {348,     75,      NO_CNCT, 22,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      312,     NO_CNCT, NO_CNCT, NO_CNCT, 224,     NO_CNCT, NO_CNCT, 17,      NO_CNCT, 59,      314,     NO_CNCT,
      244,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {156,     88,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 293,     111,
      NO_CNCT, 92,      NO_CNCT, NO_CNCT, NO_CNCT, 152,     23,      NO_CNCT, 337,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 175,     253,     NO_CNCT, 27,      NO_CNCT, NO_CNCT, 231,     49,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 267,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {25,      322,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      200,     NO_CNCT, NO_CNCT, NO_CNCT, 351,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 166,     338,     192,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {123,     217,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 142,     110,
      NO_CNCT, 176,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 76,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {311,     NO_CNCT, NO_CNCT, 251,     NO_CNCT, NO_CNCT, NO_CNCT, 265,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 94,      NO_CNCT, NO_CNCT, 81,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {22,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      322,     NO_CNCT, NO_CNCT, 277,     156,     66,      NO_CNCT, NO_CNCT, NO_CNCT, 78,      NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {176,     348,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 15,      NO_CNCT,
      NO_CNCT, 81,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 176,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 113,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 190,     NO_CNCT, 293,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 332,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 331,     NO_CNCT, 114,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {110,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 228,     NO_CNCT, 247,     116,     NO_CNCT, NO_CNCT, NO_CNCT, 190,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 47,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      286,     246,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 181,     73,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {87,      110,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 147,     258,     NO_CNCT, 204,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {89,      NO_CNCT, NO_CNCT, 65,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 155,     NO_CNCT, 244,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 30,      NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 162,     NO_CNCT, NO_CNCT, NO_CNCT, 264,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 346,     NO_CNCT, NO_CNCT, NO_CNCT, 143,     109,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {280,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      157,     236,     NO_CNCT, NO_CNCT, NO_CNCT, 113,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 18,      6,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 181,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 304,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {38,      NO_CNCT, NO_CNCT, 170,     249,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 288,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 194,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 279,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 255,     111,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 54,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {325,     NO_CNCT, 326,     NO_CNCT, 226,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 99,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 91,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 326,     NO_CNCT, 268,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {102,     NO_CNCT, NO_CNCT, NO_CNCT, 1,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 40,      NO_CNCT, 167,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 273,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 104,     NO_CNCT, NO_CNCT, NO_CNCT, 243,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 107,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {171,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 16,      NO_CNCT,
      NO_CNCT, 95,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      212,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 101,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 297,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 279,     NO_CNCT,
      NO_CNCT, 222,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {351,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      265,     NO_CNCT, 338,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      83,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 56,      304,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 141,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 101,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {60,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 320,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 112,     NO_CNCT, 54,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 100,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 210,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      195,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 268,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {135,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 15,      35,      NO_CNCT, NO_CNCT, 188,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 319,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 236,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 85,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {164,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 196,     209,     NO_CNCT,
      246,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 236,     NO_CNCT, 264,     NO_CNCT, NO_CNCT, NO_CNCT, 37,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 272,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {304,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 237,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 135,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 123,     NO_CNCT, 77,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 25,      NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 272,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {288,     NO_CNCT, NO_CNCT, NO_CNCT, 83,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      17,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 210,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 3,       NO_CNCT, 53,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 167,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT},
     {79,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 244,     NO_CNCT, 293,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 272,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT},
     {NO_CNCT, 82,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 67,      NO_CNCT, NO_CNCT, NO_CNCT, 235,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0}},

    // BG1 - lifting size index 6
    {{0,       0,       0,       0,       NO_CNCT, 0,       0,       NO_CNCT, NO_CNCT, 0,       0,       0,
      0,       0,       NO_CNCT, 0,       0,       NO_CNCT, 0,       0,       0,       0,       0,       0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {22,      NO_CNCT, 11,      124,     0,       10,      NO_CNCT, 0,       0,       2,       NO_CNCT, 16,
      60,      NO_CNCT, 0,       6,       30,      0,       NO_CNCT, 168,     NO_CNCT, 31,      105,     0,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {132,     37,      21,      NO_CNCT, 180,     4,       149,     48,      38,      122,     195,     NO_CNCT,
      NO_CNCT, 155,     28,      85,      NO_CNCT, 47,      179,     42,      66,      NO_CNCT, NO_CNCT, NO_CNCT,
      0,       0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {4,       6,       NO_CNCT, 33,      113,     NO_CNCT, 49,      21,      6,       NO_CNCT, 151,     83,
      154,     87,      5,       NO_CNCT, 92,      173,     120,     NO_CNCT, 2,       142,     0,       NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {24,      204,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {185,     100,     NO_CNCT, 24,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      65,      NO_CNCT, NO_CNCT, NO_CNCT, 207,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 161,     72,      NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {6,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 27,      NO_CNCT, NO_CNCT, NO_CNCT, 163,     50,
      NO_CNCT, 48,      NO_CNCT, NO_CNCT, NO_CNCT, 24,      38,      NO_CNCT, 91,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {145,     88,      NO_CNCT, NO_CNCT, 112,     NO_CNCT, NO_CNCT, 153,     159,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 76,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {172,     2,       NO_CNCT, 131,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      141,     NO_CNCT, NO_CNCT, NO_CNCT, 96,      NO_CNCT, NO_CNCT, 99,      NO_CNCT, 101,     35,      NO_CNCT,
      116,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {6,       10,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 145,     53,
      NO_CNCT, 201,     NO_CNCT, NO_CNCT, NO_CNCT, 4,       164,     NO_CNCT, 173,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 126,     77,      NO_CNCT, 156,     NO_CNCT, NO_CNCT, 16,      12,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 70,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {184,     194,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      123,     NO_CNCT, NO_CNCT, NO_CNCT, 16,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 104,     109,     124,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {6,       20,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 203,     153,
      NO_CNCT, 104,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 207,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {52,      NO_CNCT, NO_CNCT, 147,     NO_CNCT, NO_CNCT, NO_CNCT, 1,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 16,      NO_CNCT, NO_CNCT, 46,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {1,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      202,     NO_CNCT, NO_CNCT, 118,     130,     1,       NO_CNCT, NO_CNCT, NO_CNCT, 2,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {173,     6,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 81,      NO_CNCT,
      NO_CNCT, 182,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 53,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 46,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 88,      NO_CNCT, 198,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 160,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 122,     NO_CNCT, 182,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {91,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 184,     NO_CNCT, 30,      3,       NO_CNCT, NO_CNCT, NO_CNCT, 155,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 1,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      41,      167,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 68,      148,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {12,      6,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 166,     184,     NO_CNCT, 191,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {6,       NO_CNCT, NO_CNCT, 12,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 15,      NO_CNCT, 5,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 30,      NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 6,       NO_CNCT, NO_CNCT, NO_CNCT, 86,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 96,      NO_CNCT, NO_CNCT, NO_CNCT, 42,      199,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {44,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      58,      130,     NO_CNCT, NO_CNCT, NO_CNCT, 131,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 45,      18,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 132,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 100,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {9,       NO_CNCT, NO_CNCT, 125,     191,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 28,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 6,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 4,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 74,      16,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 28,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {21,      NO_CNCT, 142,     NO_CNCT, 192,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 197,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 98,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 140,     NO_CNCT, 22,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {4,       NO_CNCT, NO_CNCT, NO_CNCT, 1,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 40,      NO_CNCT, 93,      NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 92,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 136,     NO_CNCT, NO_CNCT, NO_CNCT, 106,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 6,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {2,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 88,      NO_CNCT,
      NO_CNCT, 112,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      20,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 4,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 49,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 125,     NO_CNCT,
      NO_CNCT, 194,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {6,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      126,     NO_CNCT, 63,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      20,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 10,      30,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 6,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 92,      NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {4,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 153,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 197,     NO_CNCT, 155,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 4,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 45,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      168,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 185,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {6,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 200,     177,     NO_CNCT, NO_CNCT, 43,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 82,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 2,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 135,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {91,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 64,      198,     NO_CNCT,
      100,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 4,       NO_CNCT, 28,      NO_CNCT, NO_CNCT, NO_CNCT, 109,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 188,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {10,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 84,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 12,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 2,       NO_CNCT, 75,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 142,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 128,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {163,     NO_CNCT, NO_CNCT, NO_CNCT, 10,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      162,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 1,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 163,     NO_CNCT, 99,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 98,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT},
     {4,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 6,       NO_CNCT, 142,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 3,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT},
     {NO_CNCT, 181,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 45,      NO_CNCT, NO_CNCT, NO_CNCT, 153,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0}},

    // BG1 - lifting size index 7
    {{135,     227,     126,     134,     NO_CNCT, 84,      83,      NO_CNCT, NO_CNCT, 53,      225,     205,
      128,     75,      NO_CNCT, 135,     217,     NO_CNCT, 220,     90,      105,     137,     1,       0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {96,      NO_CNCT, 236,     136,     221,     128,     NO_CNCT, 92,      172,     56,      NO_CNCT, 11,
      189,     NO_CNCT, 95,      85,      153,     87,      NO_CNCT, 163,     NO_CNCT, 216,     0,       0,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {189,     4,       225,     NO_CNCT, 151,     236,     117,     179,     92,      24,      68,      NO_CNCT,
      NO_CNCT, 6,       101,     33,      NO_CNCT, 96,      125,     67,      230,     NO_CNCT, NO_CNCT, NO_CNCT,
      0,       0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {128,     23,      NO_CNCT, 162,     220,     NO_CNCT, 43,      186,     96,      NO_CNCT, 1,       216,
      22,      24,      167,     NO_CNCT, 200,     32,      235,     NO_CNCT, 172,     219,     1,       NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {64,      211,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {2,       171,     NO_CNCT, 47,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      143,     NO_CNCT, NO_CNCT, NO_CNCT, 210,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 180,     180,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {199,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 22,      NO_CNCT, NO_CNCT, NO_CNCT, 23,      100,
      NO_CNCT, 92,      NO_CNCT, NO_CNCT, NO_CNCT, 207,     52,      NO_CNCT, 13,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {77,      146,     NO_CNCT, NO_CNCT, 209,     NO_CNCT, NO_CNCT, 32,      166,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 18,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {181,     105,     NO_CNCT, 141,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      223,     NO_CNCT, NO_CNCT, NO_CNCT, 177,     NO_CNCT, NO_CNCT, 145,     NO_CNCT, 199,     153,     NO_CNCT,
      38,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {169,     12,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 206,     221,
      NO_CNCT, 17,      NO_CNCT, NO_CNCT, NO_CNCT, 212,     92,      NO_CNCT, 205,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 116,     151,     NO_CNCT, 70,      NO_CNCT, NO_CNCT, 230,     115,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 84,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {45,      115,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      134,     NO_CNCT, NO_CNCT, NO_CNCT, 1,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 152,     165,     107,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {186,     215,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 124,     180,
      NO_CNCT, 98,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 80,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {220,     NO_CNCT, NO_CNCT, 185,     NO_CNCT, NO_CNCT, NO_CNCT, 154,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 178,     NO_CNCT, NO_CNCT, 150,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {124,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      144,     NO_CNCT, NO_CNCT, 182,     95,      72,      NO_CNCT, NO_CNCT, NO_CNCT, 76,      NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {39,      138,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 220,     NO_CNCT,
      NO_CNCT, 173,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 142,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 49,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 78,      NO_CNCT, 152,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 84,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 5,       NO_CNCT, 205,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {183,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 112,     NO_CNCT, 106,     219,     NO_CNCT, NO_CNCT, NO_CNCT, 129,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 183,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      215,     180,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 143,     14,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {179,     108,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 159,     138,     NO_CNCT, 196,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {77,      NO_CNCT, NO_CNCT, 187,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 203,     NO_CNCT, 167,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 130,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 197,     NO_CNCT, NO_CNCT, NO_CNCT, 122,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 215,     NO_CNCT, NO_CNCT, NO_CNCT, 65,      216,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {25,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      47,      126,     NO_CNCT, NO_CNCT, NO_CNCT, 178,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 185,     127,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 117,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 199,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {32,      NO_CNCT, NO_CNCT, 178,     2,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 156,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 58,      NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 27,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 141,     11,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 181,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {163,     NO_CNCT, 131,     NO_CNCT, 169,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 98,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 165,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 232,     NO_CNCT, 9,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {32,      NO_CNCT, NO_CNCT, NO_CNCT, 43,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 200,     NO_CNCT, 205,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 232,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 32,      NO_CNCT, NO_CNCT, NO_CNCT, 118,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 103,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {170,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 199,     NO_CNCT,
      NO_CNCT, 26,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      105,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 73,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 149,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 175,     NO_CNCT,
      NO_CNCT, 108,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {103,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      110,     NO_CNCT, 151,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      211,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 199,     132,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 172,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 65,      NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {161,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 237,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 142,     NO_CNCT, 180,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 231,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 174,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      145,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 100,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {11,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 207,     42,      NO_CNCT, NO_CNCT, 100,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 59,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 204,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 161,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {121,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 90,      26,      NO_CNCT,
      140,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 115,     NO_CNCT, 188,     NO_CNCT, NO_CNCT, NO_CNCT, 168,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 52,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {4,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 103,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 30,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 53,      NO_CNCT, 189,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 215,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 24,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {222,     NO_CNCT, NO_CNCT, NO_CNCT, 170,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      71,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 22,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 127,     NO_CNCT, 49,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 125,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT},
     {191,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 211,     NO_CNCT, 187,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 148,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT},
     {NO_CNCT, 177,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 114,     NO_CNCT, NO_CNCT, NO_CNCT, 93,      NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0}}};

/*!
 * \brief Look-up table for Base Graph BG2.
 *
 * BG2 has sizes 42x52. There are 8 possible variations according to the lifting
 * size index. Each entry corresponds to the cyclic rotation applied to the
 * (check node)--(variable node) connection when lifting the graph. Entries
 * marked as NO_CNCT mean that there is no connection between the corresponding
 * check node and variable node (in both the base graph and the lifted one).
 */
static const uint16_t BG2_matrices[NOF_LIFTSIZE][BG2M][BG2Nfull] = {
    // BG2 - lifting size index 0
    {{9,       117,     204,     26,      NO_CNCT, NO_CNCT, 189,     NO_CNCT, NO_CNCT, 205,     0,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {167,     NO_CNCT, NO_CNCT, 166,     253,     125,     226,     156,     224,     252,     NO_CNCT,
      0,       0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {81,      114,     NO_CNCT, 44,      52,      NO_CNCT, NO_CNCT, NO_CNCT, 240,     NO_CNCT, 1,
      NO_CNCT, 0,       0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 8,       58,      NO_CNCT, 158,     104,     209,     54,      18,      128,     0,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {179,     214,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      71,      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {231,     41,      NO_CNCT, NO_CNCT, NO_CNCT, 194,     NO_CNCT, 159,     NO_CNCT, NO_CNCT, NO_CNCT,
      103,     NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {155,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 228,     NO_CNCT, 45,      NO_CNCT, 28,      NO_CNCT,
      158,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 129,     NO_CNCT, NO_CNCT, NO_CNCT, 147,     NO_CNCT, 140,     NO_CNCT, NO_CNCT, NO_CNCT,
      3,       NO_CNCT, 116,     NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {142,     94,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 230,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 203,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 205,     NO_CNCT, 61,
      247,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {11,      185,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       117,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {11,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 236,     NO_CNCT, 210,     NO_CNCT,
      NO_CNCT, NO_CNCT, 56,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 63,      NO_CNCT, 111,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      14,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {83,      2,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 38,      NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 222,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 115,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 145,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      3,       NO_CNCT, 232,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {51,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 175,
      213,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 203,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 142,     NO_CNCT,
      8,       242,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 254,     NO_CNCT, NO_CNCT, NO_CNCT, 124,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      114,     64,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {220,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 194,     50,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {87,      20,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 185,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 26,      NO_CNCT, NO_CNCT, 105,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      29,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {76,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 42,      NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 210,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 222,     63,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {23,      NO_CNCT, NO_CNCT, 235,     NO_CNCT, 238,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 46,      139,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 8,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {228,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 156,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 29,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 143,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 160,     122,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {8,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 151,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 98,      101,     NO_CNCT, NO_CNCT, 135,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {18,      NO_CNCT, NO_CNCT, NO_CNCT, 28,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 71,      NO_CNCT, NO_CNCT, 240,     NO_CNCT, 9,       NO_CNCT, 84,      NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 106,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 1,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {242,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 44,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 166,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 132,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 164,     NO_CNCT, NO_CNCT, 235,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {147,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 85,      36,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 57,      NO_CNCT, NO_CNCT, NO_CNCT, 40,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      63,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {140,     NO_CNCT, 38,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 154,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 219,
      NO_CNCT, NO_CNCT, 151,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 31,      NO_CNCT, NO_CNCT, NO_CNCT, 66,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      38,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT},
     {239,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 172,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 34,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 75,
      NO_CNCT, NO_CNCT, 120,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT},
     {NO_CNCT, 129,     NO_CNCT, NO_CNCT, NO_CNCT, 229,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      118,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0}},

    // BG2 - lifting size index 1
    {{174,     97,      166,     66,      NO_CNCT, NO_CNCT, 71,      NO_CNCT, NO_CNCT, 172,     0,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {27,      NO_CNCT, NO_CNCT, 36,      48,      92,      31,      187,     185,     3,       NO_CNCT,
      0,       0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {25,      114,     NO_CNCT, 117,     110,     NO_CNCT, NO_CNCT, NO_CNCT, 114,     NO_CNCT, 1,
      NO_CNCT, 0,       0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 136,     175,     NO_CNCT, 113,     72,      123,     118,     28,      186,     0,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {72,      74,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      29,      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {10,      44,      NO_CNCT, NO_CNCT, NO_CNCT, 121,     NO_CNCT, 80,      NO_CNCT, NO_CNCT, NO_CNCT,
      48,      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {129,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 92,      NO_CNCT, 100,     NO_CNCT, 49,      NO_CNCT,
      184,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 80,      NO_CNCT, NO_CNCT, NO_CNCT, 186,     NO_CNCT, 16,      NO_CNCT, NO_CNCT, NO_CNCT,
      102,     NO_CNCT, 143,     NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {118,     70,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 152,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 28,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 132,     NO_CNCT, 185,
      178,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {59,      104,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 22,      52,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {32,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 92,      NO_CNCT, 174,     NO_CNCT,
      NO_CNCT, NO_CNCT, 154,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 39,      NO_CNCT, 93,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      11,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {49,      125,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 35,      NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 166,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 19,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 118,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      21,      NO_CNCT, 163,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {68,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 63,
      81,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 87,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 177,     NO_CNCT,
      135,     64,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 158,     NO_CNCT, NO_CNCT, NO_CNCT, 23,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      9,       6,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {186,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 6,       46,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {58,      42,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 156,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 76,      NO_CNCT, NO_CNCT, 61,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      153,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {157,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 175,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 67,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 20,      52,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {106,     NO_CNCT, NO_CNCT, 86,      NO_CNCT, 95,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 182,     153,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 64,      NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {45,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 21,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 67,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 137,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 55,      85,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {103,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 50,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 70,      111,     NO_CNCT, NO_CNCT, 168,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {110,     NO_CNCT, NO_CNCT, NO_CNCT, 17,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 120,     NO_CNCT, NO_CNCT, 154,     NO_CNCT, 52,      NO_CNCT, 56,      NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 3,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 170,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {84,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 8,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 17,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 165,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 179,     NO_CNCT, NO_CNCT, 124,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {173,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 177,     12,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 77,      NO_CNCT, NO_CNCT, NO_CNCT, 184,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      18,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {25,      NO_CNCT, 151,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 170,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 37,
      NO_CNCT, NO_CNCT, 31,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 84,      NO_CNCT, NO_CNCT, NO_CNCT, 151,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      190,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT},
     {93,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 132,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 57,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 103,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 107,
      NO_CNCT, NO_CNCT, 163,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT},
     {NO_CNCT, 147,     NO_CNCT, NO_CNCT, NO_CNCT, 7,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      60,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0}},

    // BG2 - lifting size index 2
    {{0,       0,       0,       0,       NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, 0,       0,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {137,     NO_CNCT, NO_CNCT, 124,     0,       0,       88,      0,       0,       55,      NO_CNCT,
      0,       0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {20,      94,      NO_CNCT, 99,      9,       NO_CNCT, NO_CNCT, NO_CNCT, 108,     NO_CNCT, 1,
      NO_CNCT, 0,       0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 38,      15,      NO_CNCT, 102,     146,     12,      57,      53,      46,      0,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {0,       136,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      157,     NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {0,       131,     NO_CNCT, NO_CNCT, NO_CNCT, 142,     NO_CNCT, 141,     NO_CNCT, NO_CNCT, NO_CNCT,
      64,      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 124,     NO_CNCT, 99,      NO_CNCT, 45,      NO_CNCT,
      148,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, 45,      NO_CNCT, 148,     NO_CNCT, NO_CNCT, NO_CNCT,
      96,      NO_CNCT, 78,      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {0,       65,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 87,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 97,      NO_CNCT, 51,
      85,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {0,       17,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 156,     20,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 7,       NO_CNCT, 4,       NO_CNCT,
      NO_CNCT, NO_CNCT, 2,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 0,       NO_CNCT, 113,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      48,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {0,       112,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 102,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 26,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 138,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      57,      NO_CNCT, 27,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 73,
      99,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 79,      NO_CNCT,
      111,     143,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, 24,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      109,     18,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 18,      86,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {0,       158,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 154,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 0,       NO_CNCT, NO_CNCT, 148,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      104,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 17,      NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 33,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 0,       4,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {0,       NO_CNCT, NO_CNCT, 75,      NO_CNCT, 158,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 0,       69,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 87,      NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 65,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 100,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 13,      7,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 32,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 0,       126,     NO_CNCT, NO_CNCT, 110,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {0,       NO_CNCT, NO_CNCT, NO_CNCT, 154,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, 35,      NO_CNCT, 51,      NO_CNCT, 134,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 20,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 20,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 122,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 88,      NO_CNCT, NO_CNCT, 13,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 19,      78,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, 157,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      6,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {0,       NO_CNCT, 63,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 82,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, 144,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, 93,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      19,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT},
     {0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 24,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 138,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 36,
      NO_CNCT, NO_CNCT, 143,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT},
     {NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, 2,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      55,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0}},

    // BG2 - lifting size index 3
    {{72,      110,     23,      181,     NO_CNCT, NO_CNCT, 95,      NO_CNCT, NO_CNCT, 8,       1,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {53,      NO_CNCT, NO_CNCT, 156,     115,     156,     115,     200,     29,      31,      NO_CNCT,
      0,       0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {152,     131,     NO_CNCT, 46,      191,     NO_CNCT, NO_CNCT, NO_CNCT, 91,      NO_CNCT, 0,
      NO_CNCT, 0,       0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 185,     6,       NO_CNCT, 36,      124,     124,     110,     156,     133,     1,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {200,     16,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      101,     NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {185,     138,     NO_CNCT, NO_CNCT, NO_CNCT, 170,     NO_CNCT, 219,     NO_CNCT, NO_CNCT, NO_CNCT,
      193,     NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {123,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 55,      NO_CNCT, 31,      NO_CNCT, 222,     NO_CNCT,
      209,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 103,     NO_CNCT, NO_CNCT, NO_CNCT, 13,      NO_CNCT, 105,     NO_CNCT, NO_CNCT, NO_CNCT,
      150,     NO_CNCT, 181,     NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {147,     43,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 152,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 2,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 30,      NO_CNCT, 184,
      83,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {174,     150,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 8,       56,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {99,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 138,     NO_CNCT, 110,     NO_CNCT,
      NO_CNCT, NO_CNCT, 99,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 46,      NO_CNCT, 217,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      109,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {37,      113,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 143,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 140,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 36,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 95,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      40,      NO_CNCT, 116,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {116,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 200,
      110,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 75,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 158,     NO_CNCT,
      134,     97,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 48,      NO_CNCT, NO_CNCT, NO_CNCT, 132,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      206,     2,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {68,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 16,      156,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {35,      138,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 86,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 6,       NO_CNCT, NO_CNCT, 20,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      141,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {80,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 43,      NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 81,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 49,      1,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {156,     NO_CNCT, NO_CNCT, 54,      NO_CNCT, 134,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 153,     88,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 63,      NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {211,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 94,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 90,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 6,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 221,     6,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {27,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 118,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 216,     212,     NO_CNCT, NO_CNCT, 193,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {108,     NO_CNCT, NO_CNCT, NO_CNCT, 61,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 106,     NO_CNCT, NO_CNCT, 44,      NO_CNCT, 185,     NO_CNCT, 176,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 147,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 182,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {108,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 21,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 110,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 71,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 12,      NO_CNCT, NO_CNCT, 109,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {29,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 201,     69,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 91,      NO_CNCT, NO_CNCT, NO_CNCT, 165,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      55,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {1,       NO_CNCT, 175,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 83,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 40,
      NO_CNCT, NO_CNCT, 12,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 37,      NO_CNCT, NO_CNCT, NO_CNCT, 97,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      46,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT},
     {106,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 181,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 154,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 98,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 35,
      NO_CNCT, NO_CNCT, 36,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT},
     {NO_CNCT, 120,     NO_CNCT, NO_CNCT, NO_CNCT, 101,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      81,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0}},

    // BG2 - lifting size index 4
    {{3,       26,      53,      35,      NO_CNCT, NO_CNCT, 115,     NO_CNCT, NO_CNCT, 127,     0,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {19,      NO_CNCT, NO_CNCT, 94,      104,     66,      84,      98,      69,      50,      NO_CNCT,
      0,       0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {95,      106,     NO_CNCT, 92,      110,     NO_CNCT, NO_CNCT, NO_CNCT, 111,     NO_CNCT, 1,
      NO_CNCT, 0,       0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 120,     121,     NO_CNCT, 22,      4,       73,      49,      128,     79,      0,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {42,      24,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      51,      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {40,      140,     NO_CNCT, NO_CNCT, NO_CNCT, 84,      NO_CNCT, 137,     NO_CNCT, NO_CNCT, NO_CNCT,
      71,      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {109,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 87,      NO_CNCT, 107,     NO_CNCT, 133,     NO_CNCT,
      139,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 97,      NO_CNCT, NO_CNCT, NO_CNCT, 135,     NO_CNCT, 35,      NO_CNCT, NO_CNCT, NO_CNCT,
      108,     NO_CNCT, 65,      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {70,      69,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 88,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 97,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 40,      NO_CNCT, 24,
      49,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {46,      41,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 101,     96,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {28,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 30,      NO_CNCT, 116,     NO_CNCT,
      NO_CNCT, NO_CNCT, 64,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 33,      NO_CNCT, 122,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      131,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {76,      37,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 62,      NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 47,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 143,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 51,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      130,     NO_CNCT, 97,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {139,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 96,
      128,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 48,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 9,       NO_CNCT,
      28,      8,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 120,     NO_CNCT, NO_CNCT, NO_CNCT, 43,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      65,      42,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {17,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 106,     142,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {79,      28,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 41,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 2,       NO_CNCT, NO_CNCT, 103,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      78,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {91,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 75,      NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 81,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 54,      132,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {68,      NO_CNCT, NO_CNCT, 115,     NO_CNCT, 56,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 30,      42,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 101,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {128,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 63,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 142,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 28,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 100,     133,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {13,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 10,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 106,     77,      NO_CNCT, NO_CNCT, 43,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {133,     NO_CNCT, NO_CNCT, NO_CNCT, 25,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 87,      NO_CNCT, NO_CNCT, 56,      NO_CNCT, 104,     NO_CNCT, 70,      NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 80,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 139,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {32,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 89,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 71,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 135,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 6,       NO_CNCT, NO_CNCT, 2,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {37,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 25,      114,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 60,      NO_CNCT, NO_CNCT, NO_CNCT, 137,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      93,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {121,     NO_CNCT, 129,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 26,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 97,
      NO_CNCT, NO_CNCT, 56,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 1,       NO_CNCT, NO_CNCT, NO_CNCT, 70,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      1,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT},
     {119,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 32,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 142,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 6,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 73,
      NO_CNCT, NO_CNCT, 102,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT},
     {NO_CNCT, 48,      NO_CNCT, NO_CNCT, NO_CNCT, 47,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      19,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0}},

    // BG2 - lifting size index 5
    {{156,     143,     14,      3,       NO_CNCT, NO_CNCT, 40,      NO_CNCT, NO_CNCT, 123,     0,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {17,      NO_CNCT, NO_CNCT, 65,      63,      1,       55,      37,      171,     133,     NO_CNCT,
      0,       0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {98,      168,     NO_CNCT, 107,     82,      NO_CNCT, NO_CNCT, NO_CNCT, 142,     NO_CNCT, 1,
      NO_CNCT, 0,       0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 53,      174,     NO_CNCT, 174,     127,     17,      89,      17,      105,     0,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {86,      67,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      83,      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {79,      84,      NO_CNCT, NO_CNCT, NO_CNCT, 35,      NO_CNCT, 103,     NO_CNCT, NO_CNCT, NO_CNCT,
      60,      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {47,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 154,     NO_CNCT, 10,      NO_CNCT, 155,     NO_CNCT,
      29,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 48,      NO_CNCT, NO_CNCT, NO_CNCT, 125,     NO_CNCT, 24,      NO_CNCT, NO_CNCT, NO_CNCT,
      47,      NO_CNCT, 55,      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {53,      31,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 161,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 104,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 142,     NO_CNCT, 99,
      64,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {111,     25,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 174,     23,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {91,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 175,     NO_CNCT, 24,      NO_CNCT,
      NO_CNCT, NO_CNCT, 141,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 122,     NO_CNCT, 11,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      4,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {29,      91,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 27,      NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 127,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 11,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 145,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      8,       NO_CNCT, 166,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {137,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 103,
      40,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 78,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 158,     NO_CNCT,
      17,      165,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 134,     NO_CNCT, NO_CNCT, NO_CNCT, 23,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      62,      163,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {173,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 31,      22,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {13,      135,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 145,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 128,     NO_CNCT, NO_CNCT, 52,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      173,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {156,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 166,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 40,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 18,      163,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {110,     NO_CNCT, NO_CNCT, 132,     NO_CNCT, 150,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 113,     108,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 61,      NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {72,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 136,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 36,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 38,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 53,      145,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {42,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 104,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 64,      24,      NO_CNCT, NO_CNCT, 149,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {139,     NO_CNCT, NO_CNCT, NO_CNCT, 161,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 84,      NO_CNCT, NO_CNCT, 173,     NO_CNCT, 93,      NO_CNCT, 29,      NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 117,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 148,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {116,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 73,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 142,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 105,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 137,     NO_CNCT, NO_CNCT, 29,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {11,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 41,      162,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 126,     NO_CNCT, NO_CNCT, NO_CNCT, 152,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      172,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {73,      NO_CNCT, 154,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 129,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 167,
      NO_CNCT, NO_CNCT, 38,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 112,     NO_CNCT, NO_CNCT, NO_CNCT, 7,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      19,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT},
     {109,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 6,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 105,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 160,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 156,
      NO_CNCT, NO_CNCT, 82,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT},
     {NO_CNCT, 132,     NO_CNCT, NO_CNCT, NO_CNCT, 6,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      8,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0}},

    // BG2 - lifting size index 6
    {{143,     19,      176,     165,     NO_CNCT, NO_CNCT, 196,     NO_CNCT, NO_CNCT, 13,      0,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {18,      NO_CNCT, NO_CNCT, 27,      3,       102,     185,     17,      14,      180,     NO_CNCT,
      0,       0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {126,     163,     NO_CNCT, 47,      183,     NO_CNCT, NO_CNCT, NO_CNCT, 132,     NO_CNCT, 1,
      NO_CNCT, 0,       0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 36,      48,      NO_CNCT, 18,      111,     203,     3,       191,     160,     0,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {43,      27,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      117,     NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {136,     49,      NO_CNCT, NO_CNCT, NO_CNCT, 36,      NO_CNCT, 132,     NO_CNCT, NO_CNCT, NO_CNCT,
      62,      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {7,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 34,      NO_CNCT, 198,     NO_CNCT, 168,     NO_CNCT,
      12,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 163,     NO_CNCT, NO_CNCT, NO_CNCT, 78,      NO_CNCT, 143,     NO_CNCT, NO_CNCT, NO_CNCT,
      107,     NO_CNCT, 58,      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {101,     177,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 22,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 186,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 27,      NO_CNCT, 205,
      81,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {125,     60,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 177,     51,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {39,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 29,      NO_CNCT, 35,      NO_CNCT,
      NO_CNCT, NO_CNCT, 8,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 18,      NO_CNCT, 155,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      49,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {32,      53,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 95,      NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 186,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 91,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 20,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      52,      NO_CNCT, 109,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {174,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 108,
      102,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 125,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 31,      NO_CNCT,
      54,      176,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 57,      NO_CNCT, NO_CNCT, NO_CNCT, 201,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      142,     35,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {129,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 203,     140,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {110,     124,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 52,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 196,     NO_CNCT, NO_CNCT, 35,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      114,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {10,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 122,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 23,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 202,     126,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {52,      NO_CNCT, NO_CNCT, 170,     NO_CNCT, 13,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 113,     161,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 88,      NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {197,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 194,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 164,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 172,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 49,      161,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {168,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 193,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 14,      186,     NO_CNCT, NO_CNCT, 46,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {50,      NO_CNCT, NO_CNCT, NO_CNCT, 27,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 70,      NO_CNCT, NO_CNCT, 17,      NO_CNCT, 50,      NO_CNCT, 6,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 115,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 189,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {110,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 163,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 163,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 173,     NO_CNCT, NO_CNCT, 179,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {197,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 191,     193,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 157,     NO_CNCT, NO_CNCT, NO_CNCT, 167,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      181,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {197,     NO_CNCT, 167,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 179,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 181,
      NO_CNCT, NO_CNCT, 193,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 157,     NO_CNCT, NO_CNCT, NO_CNCT, 173,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      191,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT},
     {181,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 157,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 173,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 193,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 163,
      NO_CNCT, NO_CNCT, 179,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT},
     {NO_CNCT, 191,     NO_CNCT, NO_CNCT, NO_CNCT, 197,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      167,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0}},

    // BG2 - lifting size index 7
    {{145,     131,     71,      21,      NO_CNCT, NO_CNCT, 23,      NO_CNCT, NO_CNCT, 112,     1,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {142,     NO_CNCT, NO_CNCT, 174,     183,     27,      96,      23,      9,       167,     NO_CNCT,
      0,       0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {74,      31,      NO_CNCT, 3,       53,      NO_CNCT, NO_CNCT, NO_CNCT, 155,     NO_CNCT, 0,
      NO_CNCT, 0,       0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 239,     171,     NO_CNCT, 95,      110,     159,     199,     43,      75,      1,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {29,      140,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      180,     NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {121,     41,      NO_CNCT, NO_CNCT, NO_CNCT, 169,     NO_CNCT, 88,      NO_CNCT, NO_CNCT, NO_CNCT,
      207,     NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {137,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 72,      NO_CNCT, 172,     NO_CNCT, 124,     NO_CNCT,
      56,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 86,      NO_CNCT, NO_CNCT, NO_CNCT, 186,     NO_CNCT, 87,      NO_CNCT, NO_CNCT, NO_CNCT,
      172,     NO_CNCT, 154,     NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {176,     169,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 225,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 167,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 238,     NO_CNCT, 48,
      68,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {38,      217,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 208,     232,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {178,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 214,     NO_CNCT, 168,     NO_CNCT,
      NO_CNCT, NO_CNCT, 51,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 124,     NO_CNCT, 122,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      72,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {48,      57,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 167,     NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 219,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 82,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 232,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      204,     NO_CNCT, 162,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {38,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 217,
      157,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 170,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 23,      NO_CNCT,
      175,     202,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 196,     NO_CNCT, NO_CNCT, NO_CNCT, 173,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      195,     218,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {128,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 211,     210,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {39,      84,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 88,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 117,     NO_CNCT, NO_CNCT, 227,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      6,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {238,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 13,      NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 11,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 195,     44,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {5,       NO_CNCT, NO_CNCT, 94,      NO_CNCT, 111,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 81,      19,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 130,     NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {66,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 95,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 146,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 66,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 190,     86,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {64,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 181,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 7,       144,     NO_CNCT, NO_CNCT, 16,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {25,      NO_CNCT, NO_CNCT, NO_CNCT, 57,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 37,      NO_CNCT, NO_CNCT, 139,     NO_CNCT, 221,     NO_CNCT, 17,      NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 201,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 46,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {179,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 14,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 116,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 46,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 2,       NO_CNCT, NO_CNCT, 106,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {184,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 135,     141,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 85,      NO_CNCT, NO_CNCT, NO_CNCT, 225,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      175,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {178,     NO_CNCT, 112,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 106,     NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 154,
      NO_CNCT, NO_CNCT, 114,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT},
     {NO_CNCT, 42,      NO_CNCT, NO_CNCT, NO_CNCT, 41,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      105,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT, NO_CNCT},
     {167,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 45,      NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, 189,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT, NO_CNCT},
     {NO_CNCT, NO_CNCT, 78,      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 67,
      NO_CNCT, NO_CNCT, 180,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0,       NO_CNCT},
     {NO_CNCT, 53,      NO_CNCT, NO_CNCT, NO_CNCT, 215,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      230,     NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT,
      NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, NO_CNCT, 0}}};

/*!
 * \brief Look-up table: BG1 connection indices.
 *
 * For each check node, the corresponding row contains the indices of the
 * variable nodes connected to it.
 */
static const int8_t BG1_positions[BG1M][MAX_CNCT] = {
    {0, 1, 2, 3, 5, 6, 9, 10, 11, 12, 13, 15, 16, 18, 19, 20, 21, 22, 23, -1},
    {0, 2, 3, 4, 5, 7, 8, 9, 11, 12, 14, 15, 16, 17, 19, 21, 22, 23, 24, -1},
    {0, 1, 2, 4, 5, 6, 7, 8, 9, 10, 13, 14, 15, 17, 18, 19, 20, 24, 25, -1},
    {0, 1, 3, 4, 6, 7, 8, 10, 11, 12, 13, 14, 16, 17, 18, 20, 21, 22, 25, -1},
    {0, 1, 26, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 1, 3, 12, 16, 21, 22, 27, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 6, 10, 11, 13, 17, 18, 20, 28, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 1, 4, 7, 8, 14, 29, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 1, 3, 12, 16, 19, 21, 22, 24, 30, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 1, 10, 11, 13, 17, 18, 20, 31, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 4, 7, 8, 14, 32, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 1, 12, 16, 21, 22, 23, 33, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 1, 10, 11, 13, 18, 34, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 3, 7, 20, 23, 35, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 12, 15, 16, 17, 21, 36, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 1, 10, 13, 18, 25, 37, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 3, 11, 20, 22, 38, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 14, 16, 17, 21, 39, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 12, 13, 18, 19, 40, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 1, 7, 8, 10, 41, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 3, 9, 11, 22, 42, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 5, 16, 20, 21, 43, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 12, 13, 17, 44, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 10, 18, 45, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 3, 4, 11, 22, 46, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 6, 7, 14, 47, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 2, 4, 15, 48, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 6, 8, 49, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 4, 19, 21, 50, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 14, 18, 25, 51, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 10, 13, 24, 52, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 7, 22, 25, 53, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 12, 14, 24, 54, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 11, 21, 55, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 7, 15, 17, 56, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 6, 12, 22, 57, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 14, 15, 18, 58, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 13, 23, 59, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 9, 10, 12, 60, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 3, 7, 19, 61, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 8, 17, 62, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 3, 9, 18, 63, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 4, 24, 64, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 16, 18, 25, 65, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 7, 9, 22, 66, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 6, 10, 67, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1}};

/*!
 * \brief Look-up table: BG2 connection indices.
 *
 * For each check node, the corresponding row contains the indices of the
 * variable nodes connected to it.
 */
static const int8_t BG2_positions[BG2M][MAX_CNCT] = {
    {0, 1, 2, 3, 6, 9, 10, 11, -1, -1, -1},      {0, 3, 4, 5, 6, 7, 8, 9, 11, 12, -1},
    {0, 1, 3, 4, 8, 10, 12, 13, -1, -1, -1},     {1, 2, 4, 5, 6, 7, 8, 9, 10, 13, -1},
    {0, 1, 11, 14, -1, -1, -1, -1, -1, -1, -1},  {0, 1, 5, 7, 11, 15, -1, -1, -1, -1, -1},
    {0, 5, 7, 9, 11, 16, -1, -1, -1, -1, -1},    {1, 5, 7, 11, 13, 17, -1, -1, -1, -1, -1},
    {0, 1, 12, 18, -1, -1, -1, -1, -1, -1, -1},  {1, 8, 10, 11, 19, -1, -1, -1, -1, -1, -1},
    {0, 1, 6, 7, 20, -1, -1, -1, -1, -1, -1},    {0, 7, 9, 13, 21, -1, -1, -1, -1, -1, -1},
    {1, 3, 11, 22, -1, -1, -1, -1, -1, -1, -1},  {0, 1, 8, 13, 23, -1, -1, -1, -1, -1, -1},
    {1, 6, 11, 13, 24, -1, -1, -1, -1, -1, -1},  {0, 10, 11, 25, -1, -1, -1, -1, -1, -1, -1},
    {1, 9, 11, 12, 26, -1, -1, -1, -1, -1, -1},  {1, 5, 11, 12, 27, -1, -1, -1, -1, -1, -1},
    {0, 6, 7, 28, -1, -1, -1, -1, -1, -1, -1},   {0, 1, 10, 29, -1, -1, -1, -1, -1, -1, -1},
    {1, 4, 11, 30, -1, -1, -1, -1, -1, -1, -1},  {0, 8, 13, 31, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 32, -1, -1, -1, -1, -1, -1, -1, -1},  {0, 3, 5, 33, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 9, 34, -1, -1, -1, -1, -1, -1, -1},   {0, 5, 35, -1, -1, -1, -1, -1, -1, -1, -1},
    {2, 7, 12, 13, 36, -1, -1, -1, -1, -1, -1},  {0, 6, 37, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 2, 5, 38, -1, -1, -1, -1, -1, -1, -1},   {0, 4, 39, -1, -1, -1, -1, -1, -1, -1, -1},
    {2, 5, 7, 9, 40, -1, -1, -1, -1, -1, -1},    {1, 13, 41, -1, -1, -1, -1, -1, -1, -1, -1},
    {0, 5, 12, 42, -1, -1, -1, -1, -1, -1, -1},  {2, 7, 10, 43, -1, -1, -1, -1, -1, -1, -1},
    {0, 12, 13, 44, -1, -1, -1, -1, -1, -1, -1}, {1, 5, 11, 45, -1, -1, -1, -1, -1, -1, -1},
    {0, 2, 7, 46, -1, -1, -1, -1, -1, -1, -1},   {10, 13, 47, -1, -1, -1, -1, -1, -1, -1, -1},
    {1, 5, 11, 48, -1, -1, -1, -1, -1, -1, -1},  {0, 7, 12, 49, -1, -1, -1, -1, -1, -1, -1},
    {2, 10, 13, 50, -1, -1, -1, -1, -1, -1, -1}, {1, 5, 11, 51, -1, -1, -1, -1, -1, -1, -1}};

/*!
 * Returns a pointer to the desired variation of the selected base graph.
 */
static inline const uint16_t* get_cnct_matrix(srsran_basegraph_t bg, uint8_t ls)
{
  return (bg == BG1 ? (const uint16_t*)BG1_matrices[ls] : (const uint16_t*)BG2_matrices[ls]);
}

int create_compact_pcm(uint16_t* pcm, int8_t (*positions)[MAX_CNCT], srsran_basegraph_t bg, uint16_t ls)
{
  uint8_t  ls_index = 0;
  uint16_t value    = 0;

  const struct {
    uint8_t M[2];
    uint8_t Nfull[2];
  } BG_size = {{BG1M, BG2M}, {BG1Nfull, BG2Nfull}};
  uint8_t M = BG_size.M[bg];
  uint8_t N = BG_size.Nfull[bg];

  ls_index = get_ls_index(ls);
  if (ls_index == VOID_LIFTSIZE) {
    ERROR("Invalid lifting size %d", ls);
    return -1;
  }
  const uint16_t* tmp = get_cnct_matrix(bg, ls_index);

  const int8_t(*BGpositions)[MAX_CNCT] = (bg == BG1) ? BG1_positions : BG2_positions;
  for (int i = 0; i < M; i++) {
    for (int j = 0; j < N; j++) {
      value              = *(tmp + i * N + j);
      *(pcm + i * N + j) = (value == NO_CNCT ? NO_CNCT : value % ls);
    }
  }

  if (positions != NULL) {
    for (int i = 0; i < M; i++) {
      for (int j = 0; j < MAX_CNCT; j++) {
        positions[i][j] = BGpositions[i][j];
      }
    }
  }

  return 0;
}
